/*******************************************************************
 *
 *	File:		Video.cpp
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

extern "C" {
#include "driver.h"
}

#include <stdio.h>
#include <math.h>
#include <gdi.h>
#include "video.h"
#include "emame.h"


const KOneSecondInMicroSeconds =	1000 * 1000;

static TInt sThrottle				= 1;	// bool option, default: true
static TInt sManualDiffFrames		= 0;


/***********************************************
 *
 * BITMAP
 *
 ***********************************************/

CMameBitmap::CMameBitmap()
{
}

CMameBitmap* CMameBitmap::NewL(TInt aWidth, TInt aHeight, TDisplayMode aDisplayMode)
{
	CMameBitmap*	self = new(ELeave) CMameBitmap;
	CleanupStack::PushL(self);
	self->ConstructL(aWidth, aHeight, aDisplayMode);
	CleanupStack::Pop();
	return self;
}

CMameBitmap::~CMameBitmap()
{
	delete iBitmapContext;
	iBitmapContext = NULL;
	delete iBitmapDevice;
	iBitmapDevice = NULL;
	delete iBitmap;
	iBitmap = NULL;
	if (iOsdBitmap.line)
	{
		User::Free(iOsdBitmap.line);
		iOsdBitmap.line = NULL;
	}
}


TDisplayMode CMameBitmap::GetDisplayMode(TInt aBpp, TBool aColor)
{
	if (aBpp == 1)
		return EGray2;
	else if (aBpp == 2)
		return EGray4;
	else if (aBpp == 4)
		return aColor ? EColor16 : EGray16;
	else if (aBpp == 8)
		return aColor ? EColor256 : EGray256;
	else if (aBpp == 16)
		return EColor64K;
	else if (aBpp == 12)
		return EColor4K;
	else if (aBpp == 24)
		return EColor16M;
	else if (aBpp == 32)
		return ERgb;

	ASSERT(EFalse);
	return ENone;
}


/*
 * GetBpp returns the number of bits we need per pixel in the given
 * display mode. It is different from GetColorDepth in that GetBpp can be
 * used to calculate byte offsets in a bitmap/framebuffer.
 * For example, GetBpp(EColor4K) returns 16 because each pixel requires
 * 16 bits (of which only the least significant 12 bits are used).
 * GetColorDepth(EColor4K) returns 12.
 */
TInt CMameBitmap::GetBpp(TDisplayMode aDisplayMode)
{
	const TInt bpp[] =
		{
			-1,
			1,
			2,
			4,
			8,
			4,
			8,
			16,
			32,
			32,
			16
		};
	return bpp[aDisplayMode];
}

TInt CMameBitmap::GetColorDepth(TDisplayMode aDisplayMode)
{
	const TInt bpp[] =
		{
			-1,
			1,
			2,
			4,
			8,
			4,
			8,
			16,
			24,
			32,
			12
		};
	return bpp[aDisplayMode];
}


void CMameBitmap::ConstructL(TInt aWidth, TInt aHeight, TDisplayMode aDisplayMode)
{
	iBitmap = new (ELeave) CMameFbsBitmap;
	iBitmap->Create(TSize(aWidth, aHeight), aDisplayMode);

	iBitmapDevice = CFbsBitmapDevice::NewL(iBitmap);
	iBitmapDevice->CreateContext(iBitmapContext);

		/*
		 * Setup OsdBitmap as used by the MAME core
		 */
	iOsdBitmap.width	= aWidth;
	iOsdBitmap.height	= aHeight;
	iOsdBitmap.depth	= 8; // GetColorDepth(aDisplayMode);
	iOsdBitmap._private = (TAny *) this;
	iOsdBitmap.line		= (TUint8 **) User::Alloc(aHeight * sizeof(TUint8 *));
	User::LeaveIfNull(iOsdBitmap.line);

	TUint32*			dataAddress = iBitmap->DataAddress();
	CBitwiseBitmap*		bitwiseBitmap =iBitmap->MameAddress();
	for (TInt i=0 ; i<aHeight ; i++)
		iOsdBitmap.line[i] = (TUint8 *) bitwiseBitmap->ScanLineAddress(dataAddress, i);

		/*
		 * Setup iOsdVisibleBitmap, it's mapped as a rect into iOsdBitmap
		 */
	iOsdVisibleBitmap.width		= 0;
	iOsdVisibleBitmap.height	= 0;
	iOsdVisibleBitmap.depth		= 8; // GetColorDepth(aDisplayMode);
	iOsdVisibleBitmap._private	= (TAny *) this;
	iOsdVisibleBitmap.line		= NULL;

	SetVisibleAreaL(TRect(0, 0, aWidth, aHeight));
}

void CMameBitmap::SetVisibleAreaL(const TRect& aVisibleArea)
{
	TInt	width = aVisibleArea.Size().iWidth;
	TInt	height = aVisibleArea.Size().iHeight;

	iOsdVisibleBitmap.width		= width;
	iOsdVisibleBitmap.height	= height;

	if (iOsdVisibleBitmap.line)
	{
		User::Free(iOsdVisibleBitmap.line);
		iOsdVisibleBitmap.line = NULL;
	}
	iOsdVisibleBitmap.line = (TUint8 **) User::Alloc(height * sizeof(TUint8 *));;
	for (TInt h=0 ; h<height ; h++)
		iOsdVisibleBitmap.line[h] = iOsdBitmap.line[h + aVisibleArea.iTl.iY] + ((aVisibleArea.iTl.iX * iOsdVisibleBitmap.depth) / 8);
}




/***********************************************
 *
 * SCREEN
 *
 ***********************************************/

CMameScreen::CMameScreen(const TScreenInfoV01& aScreenInfo, TDisplayMode aDisplayMode)
{
	iHasFrameBuffer		= aScreenInfo.iScreenAddressValid;
	iFrameBuffer		= iHasFrameBuffer ? (TUint8*) aScreenInfo.iScreenAddress : NULL;
	iBytesPerScanLine	= (aScreenInfo.iScreenSize.iWidth * CMameBitmap::GetBpp(aDisplayMode)) / 8;
	iScreenSize			= aScreenInfo.iScreenSize; 
	iDisplayMode		= aDisplayMode;

#ifndef __WINS__
#ifndef __ER6__
		/*
		 * Hmm, I've just checked out the SA1100 spec. It seems that for 8bpp displays
		 * there is a 512 palette table at the beginning of the frame buffer.
		 * This code assumes that all ER5 target build with 8bpp are SA1100 machines. 
		 * I only know 2: Series 7 and Netbook.
		 */
	if (CMameBitmap::GetBpp(aDisplayMode) == 8)
		iFrameBuffer += 512;
#endif
#endif

}

CMameScreen::~CMameScreen()
{
	User::Free(iScanLine);
}


void CMameScreen::ConstructL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps)
{
		/*
		 * game throttling
		 */
	TTimeIntervalMicroSeconds32	period;
	UserHal::TickPeriod(period);
	iPeriod				= period.Int();
	iVideoFps			= aFps;
	iDoubleFrame		= -1;
	iRealFrameCount		= 0;
	iGameFrameCount		= 0;
	iTicksGameStart		= 0;

	iWidth			= iGameWidth	= aWidth;
	iHeight			= iGameHeight	= aHeight;
	iGameDepth		= aDepth;
	iOrientation	= aOrientation;

	iScanLine = (TUint8*) User::AllocL(iBytesPerScanLine);
	Mem::FillZ(iScanLine, iBytesPerScanLine);

	SetupScaling();
	SetupPosition();

	SetupScreenAccess();

	iBytesPerBitmapLine = (iWidth * iGameDepth) / 8;
}

void CMameScreen::SetWantScaling(TBool aWantScaling)
{
	iWantScaling = aWantScaling;
}

void CMameScreen::SetWantDoubleBitmap(TBool aWantDoubleBitmap)
{
	iWantDoubleBitmap = aWantDoubleBitmap;
}


void CMameScreen::SetBitmapPosition(const TPoint& aPosition)
{
	TInt	bpp = CMameBitmap::GetBpp(iDisplayMode);
	iBitmapPosition = aPosition;
	iBitmapStartAddress = iFrameBuffer + ((aPosition.iX * bpp) / 8);
	iBitmapStartAddress += (aPosition.iY * iBytesPerScanLine);
}



void CMameScreen::SetupScreenAccess()
{
	if (iHasFrameBuffer)
	{
		if ((iPixelSkipX == -1) && (iPixelSkipY == -1))
		{
			if (iDoubleBitmap)
			{
				const TBitmapDrawer	bitmapDrawers[] =
					{
						NULL,
						NULL,											// Gray2
						NULL,											// Gray4
						NULL,											// Gray16
						&CMameScreen::DirectDrawDoubleBitmap8bppTo8bpp,	// Gray256
						NULL,											// Color16
						&CMameScreen::DirectDrawDoubleBitmap8bppTo8bpp,	// Color256
						NULL,											// Color64K
						NULL,											// Color16M
						NULL,											// Rgb
						NULL											// Color4K
					};
				iBitmapDrawer = bitmapDrawers[iDisplayMode];
				iHasFrameBuffer = (iBitmapDrawer != NULL);
			}
			else
			{
				const TBitmapDrawer	bitmapDrawers[] =
					{
						NULL,
						NULL,										// Gray2
						&CMameScreen::DirectDrawBitmap8bppTo2bpp,	// Gray4
						&CMameScreen::DirectDrawBitmap8bppTo4bpp,	// Gray16
						&CMameScreen::DirectDrawBitmap8bppTo8bpp,	// Gray256
						&CMameScreen::DirectDrawBitmap8bppTo4bpp,	// Color16
						&CMameScreen::DirectDrawBitmap8bppTo8bpp,	// Color256
						NULL,										// Color64K
						NULL,										// Color16M
						NULL,										// Rgb
						&CMameScreen::DirectDrawBitmap8bppTo12bpp	// Color4K
					};
				iBitmapDrawer = bitmapDrawers[iDisplayMode];
				iHasFrameBuffer = (iBitmapDrawer != NULL);
			}
		}
		else
		{
			if (iDoubleBitmap)
			{
				const TScaledScanLineDrawer	scaledScanLineDrawers[] =
					{
						NULL,
						NULL,												// Gray2
						NULL,												// Gray4
						NULL,												// Gray16
						&CMameScreen::DrawDoubleScaledScanLine8bppTo8bpp,	// Gray256
						NULL,												// Color16
						&CMameScreen::DrawDoubleScaledScanLine8bppTo8bpp,	// Color256
						NULL,												// Color64K
						NULL,												// Color16M
						NULL,												// Rgb
						NULL,												// Color4K
					};
				iBitmapDrawer = &CMameScreen::DoubleScaledDrawBitmap;
				iScaledScanLineDrawer = scaledScanLineDrawers[iDisplayMode];
				iHasFrameBuffer = (iScaledScanLineDrawer != NULL);
			}
			else
			{
				const TScaledScanLineDrawer	scaledScanLineDrawers[] =
					{
						NULL,
						NULL,												// Gray2
						DrawScaledScanLine8bppTo2bpp,						// Gray4
						DrawScaledScanLine8bppTo4bpp,						// Gray16
						NULL,												// Gray256
						NULL,												// Color16
						NULL,												// Color256
						NULL,												// Color64K
						NULL,												// Color16M
						NULL,												// Rgb
						DrawScaledScanLine8bppTo12bpp,						// Color4K
					};
				iBitmapDrawer = &CMameScreen::ScaledDrawBitmap;
				iScaledScanLineDrawer = scaledScanLineDrawers[iDisplayMode];
				iHasFrameBuffer = (iScaledScanLineDrawer != NULL);
			}
		}
	}
}

#define KMaxDoubleFrame 5

static sDoubleFrame[KMaxDoubleFrame];
static sDoubleFrameCount[KMaxDoubleFrame];


TInt CMameScreen::SkipThisFrame()
{
	return 0;
}

/*
 * Implementing throttling isn't that easy in EPOC because EPOC timers aren't that 
 * fine grained. The smallest period you can sleep is roughly 40ms and that isn't
 * good enough. Let's call the frame rate at which the hardware is capable of running 
 * "real frame rate" and the rate at which the game should run "game frame rate". 
 * 
 * An example. Suppose the game frame rate is 50 fps and the real frame rate is 60 fps. 
 * 50fps means that each frame should take 1000/50 = 20ms and at 60 fps, each frame 
 * only takes 1000/60 = 16.67 ms. So, in this case we should sleep for 2.33 ms inbetween 
 * drawing each frame to get the correct 50fps.
 *
 * Since we cannot sleep, what we'll do instead is take advantage of a known factor:
 * the time it takes to draw a frame. In the above example, if we run at 60 fps then
 * if we draw 10 frames twice in each second, the games runs at the correct speed
 * of 50 fps as well.
 */
TInt CMameScreen::DoubleFrame(TInt aShowFps, struct osd_bitmap *bitmap)
{
	TInt	doubleFrame = 1;

		/*
		 * Initialise on first tick
		 */
	if (iTicksGameStart == 0)
	{
		iTotalGameFrames = iTotalRealFrames = 0;
		iTicksGameStart = User::TickCount();
		iPrevTicks = iTicksGameStart;
		iDoubleFrame = -1;
		Mem::Fill(sDoubleFrame, KMaxDoubleFrame * sizeof(TInt), 0);		// set to -1
		sDoubleFrameCount[0] = 1;
		sDoubleFrameCount[1] = 1;
		sDoubleFrameCount[2] = 1;
		sDoubleFrameCount[3] = 1;
		sDoubleFrameCount[4] = 1;

		iDiffFrames = 0;
		iRealFps	= 0;
		iGameFps	= 0;

		sManualDiffFrames		= 0;
	}
	if (sThrottle)
	{
		if (iGameFrameCount == iVideoFps)
		{
			TInt ticks = User::TickCount();
			TInt diff = ticks - iPrevTicks;
			iPrevTicks = ticks;

			TInt time = diff * iPeriod;
			if (!time)
				time = 1;
			iRealFps = (iRealFrameCount * KOneSecondInMicroSeconds) / time;
			iGameFps = (iGameFrameCount * KOneSecondInMicroSeconds) / time;

			if (time < KOneSecondInMicroSeconds)
				iDiffFrames = iRealFps - iVideoFps;
			else
				iDiffFrames -= (iVideoFps - iGameFps) / 2;

			TInt diffFrames = iDiffFrames;

			if (sManualDiffFrames > 0)
				diffFrames = sManualDiffFrames;

			Mem::Fill(sDoubleFrame, KMaxDoubleFrame * sizeof(TInt), 0xff);
			sDoubleFrameCount[0] = 1;
			sDoubleFrameCount[1] = 1;
			sDoubleFrameCount[2] = 1;
			sDoubleFrameCount[3] = 1;
			sDoubleFrameCount[4] = 1;
			for (TInt i=0 ; i<KMaxDoubleFrame && diffFrames; i++)
			{
				sDoubleFrame[i] = iVideoFps / diffFrames;
				if (sDoubleFrame[i] == 0) sDoubleFrame[i]=1;
				if (iVideoFps / sDoubleFrame[i] > diffFrames + 2)
					sDoubleFrame[i]++;
				if (sDoubleFrame[i] == 0)
					sDoubleFrame[i] = 1;
				diffFrames -= iVideoFps / sDoubleFrame[i];
			}
//			RDebug::Print(HERE("diff=%d, %d %d %d %d %d"), iDiffFrames, sDoubleFrame[0], sDoubleFrame[1], sDoubleFrame[2], sDoubleFrame[3], sDoubleFrame[4]);

			iGameFrameCount = 0;
			iRealFrameCount = 0;
		}
	

		for (TInt i=0 ; i<KMaxDoubleFrame ; i++)
		{
			if (sDoubleFrameCount[i] == sDoubleFrame[i])
			{
				doubleFrame++;
				sDoubleFrameCount[i] = 1;
				iRealFrameCount++;
				iTotalRealFrames++;
			}
			else
				sDoubleFrameCount[i]++;
		}

		iGameFrameCount++;
		iRealFrameCount++;
	}
	iTotalRealFrames++;
	iTotalGameFrames++;

	if (aShowFps)
	{
		static	sTotalRealFrames=0;
		static	sTotalGameFrames=0;
		static	sTotalGameFps=0;
		static	sTotalRealFps=0;
		static	sTotalTicks=0;

		char	fps[64];
		TInt	ticks = User::TickCount() - iTicksGameStart;
		TInt	seconds = (ticks * iPeriod) / (KOneSecondInMicroSeconds);

		if (seconds == 0)
			seconds = 1;

		if (seconds > 5)
		{
			sTotalRealFrames+=iTotalRealFrames;
			sTotalGameFrames+=iTotalGameFrames;
			sTotalTicks+=ticks;
			TInt	totSeconds = (sTotalTicks * iPeriod) / (KOneSecondInMicroSeconds);
			sTotalGameFps = sTotalGameFrames / totSeconds;
			sTotalRealFps = sTotalRealFrames / totSeconds;

			iTotalRealFrames=0;
			iTotalGameFrames=0;
			iTicksGameStart=User::TickCount();
		}

		if (sManualDiffFrames > 0)
			sprintf(fps, "manual: g: %d, r: %d, (%d, %d) %d", iGameFps, iRealFps, sTotalGameFps, sTotalRealFps, sManualDiffFrames);
		else
			sprintf(fps, "auto: g: %d, r: %d, (%d, %d)", iGameFps, iRealFps, sTotalGameFps, sTotalRealFps);

		ui_text(bitmap, fps, 4, 0);
	}

	return doubleFrame;
}




void CMameScreen::SetupPosition()
{
	iPosition = TPoint(0, 0);

	switch (iOrientation)
	{
		case EOrientationRotLeft:
			printf("SetupPosition: EOrientationRotLeft\n");
			iPosition = TPoint(0, 0);
			if ((iHeight +10) < iScreenSize.iHeight)
			{
				iPosition.iY = (iScreenSize.iHeight - iHeight) / 2;
			}
			break;
		case EOrientationRotRight:
			iPosition = TPoint(iScreenSize.iWidth - iWidth, 0);
			if ((iHeight +10) < iScreenSize.iHeight)
			{
				iPosition.iY = (iScreenSize.iHeight - iHeight) / 2;
			}
			break;
	
		default:
			if ((iWidth +10) < iScreenSize.iWidth)
			{
				iPosition.iX = (iScreenSize.iWidth - iWidth) / 2;
			}

			if ((iHeight +10) < iScreenSize.iHeight)
			{
				iPosition.iY = (iScreenSize.iHeight - iHeight) / 2;
			}
			break;
	}
	SetBitmapPosition(iPosition);
}

void CMameScreen::SetupScaling()
{
	iPixelSkipX = -1;
	iPixelSkipY = -1;
	iDoubleBitmap = EFalse;

	if (iWantScaling)
	{
		float	scaleX = ((float) iScreenSize.iWidth) / ((float) iGameWidth);
		float	scaleY = ((float) iScreenSize.iHeight) / ((float) iGameHeight);
		if ((scaleX < 1) || (scaleY < 1))
		{
			float scale = (scaleX < scaleY) ? scaleX : scaleY;
			iWidth = scale * iGameWidth;
			iHeight = scale * iGameHeight;

			iPixelSkipX = ceil (((float) iGameWidth) / (((float) iGameWidth) - ((float)iWidth)));
			iPixelSkipY = ceil (((float) iGameHeight) / (((float) iGameHeight) - ((float) iHeight)));

			iPixelSkipY--;	// we start counting at zero
			iPixelSkipX--;
		}
		else
		{
			TInt doubleWidth = iGameWidth * 2;
			TInt doubleHeight = iGameHeight * 2;

			float	scaleX = ((float) iScreenSize.iWidth) / ((float) doubleWidth);
			float	scaleY = ((float) iScreenSize.iHeight) / ((float) doubleHeight);
			if (((scaleX < 1) || (scaleY < 1)) && iWantDoubleBitmap)
			{
				iDoubleBitmap = ETrue;
				if ((scaleX > 0.66) && (scaleY > 0.66))
				{
					float scale = (scaleX < scaleY) ? scaleX : scaleY;
					iWidth = scale * doubleWidth;
					iHeight = scale * doubleHeight;

					iPixelSkipX = ceil (((float) doubleWidth) / (((float) doubleWidth) - ((float)iWidth)));
					iPixelSkipY = ceil (((float) doubleHeight) / (((float) doubleHeight) - ((float) iHeight)));

					iPixelSkipY--;
					iPixelSkipX--;
				}
				else
				{
					iDoubleBitmap = EFalse;
				}
			}
		}
	}
	else
	{
		iWantPanning = ETrue;		
	}
	iGameWidth-=4;
	iGameHeight-=4;
}



void CMameScreen::AllocateColors(TUint totalcolors, const UINT8 *palette, UINT16 *pens, TInt modifiable, const UINT8 *debug_palette,UINT16 *debug_pens)
{
	if (CMameBitmap::GetColorDepth(iDisplayMode) <= 8)
		AllocateEpocColors(totalcolors, palette, pens, modifiable, debug_palette, debug_pens);
	else
		AllocateMameColors(totalcolors, palette, pens, modifiable, debug_palette, debug_pens);
}

void CMameScreen::AllocateEpocColors(TUint totalcolors, const UINT8 *palette, UINT16 *pens, TInt modifiable, const UINT8 *debug_palette,UINT16 *debug_pens)
{
		/*
		 * We're not capable in ER5 to set a palette. Instead, we find the palette index
		 * which matches the requested color best. 
		 */
	CPalette*	gdiPalette = CPalette::NewDefaultL(iDisplayMode);
	CleanupStack::PushL(gdiPalette);
	for (TUint i = 0; i < totalcolors; i++)
	{
		TRgb	color(palette[3 * i], palette[3 * i + 1], palette[3 * i + 2]);
		
		pens[i] = gdiPalette->NearestIndex(color);

			/*
			 * For low color displays make sure that only black is mapped to black
			 */
		if (CMameBitmap::GetColorDepth(iDisplayMode) == 2)
		{
			if ((color != KRgbBlack) && ((pens[i] == 0) || (pens[i] == 1)))
				pens[i] = 2;
		}
		else if (CMameBitmap::GetColorDepth(iDisplayMode) == 4)
		{
			if ((color != KRgbBlack) && (pens[i] < 5))
				pens[i] = 5;
		}
	}

	Machine->uifont->colortable[0] = gdiPalette->NearestIndex(KRgbBlack);
	Machine->uifont->colortable[1] = gdiPalette->NearestIndex(KRgbWhite);
	Machine->uifont->colortable[2] = gdiPalette->NearestIndex(KRgbWhite);
	Machine->uifont->colortable[3] = gdiPalette->NearestIndex(KRgbBlack);
	CleanupStack::PopAndDestroy();
}


TUint CMameScreen::GetColorValue(const TRgb& aColor, const TDisplayMode& aDisplayMode)
{
	switch (aDisplayMode)
	{
		case EColor4K:
			return aColor.Color4K();
			break;
		case EColor16M:
			return aColor.Color16M();
			break;
		case EColor64K:
			return aColor.Color64K();
			break;
		case ERgb:
			return aColor.Value();
			break;
		default:
			ASSERT(EFalse);
	}
	return 0;
}

TInt CMameScreen::AllocateMamePen(const TRgb& aColor)
{
	TUint	colorValue = GetColorValue(aColor, iDisplayMode);
	for (TInt i=0 ; i<iMamePaletteMaxIndex; i++)
		if (iMamePalette[i] == colorValue)
			return i;
	ASSERT(iMamePaletteMaxIndex<KSizeMameColorPalette-1);
	iMamePalette[iMamePaletteMaxIndex++] = colorValue;
	return (iMamePaletteMaxIndex-1);
}


void CMameScreen::AllocateMameColors(TUint totalcolors, const UINT8 *palette, UINT16 *pens, TInt modifiable, const UINT8 *debug_palette,UINT16 *debug_pens)
{
	for (TUint i = 0; i < totalcolors; i++)
	{
		TRgb	color(palette[3 * i], palette[3 * i + 1], palette[3 * i + 2]);
		pens[i] = AllocateMamePen(color);
	}

	Machine->uifont->colortable[0] = AllocateMamePen(KRgbBlack);
	Machine->uifont->colortable[1] = AllocateMamePen(KRgbWhite);
	Machine->uifont->colortable[2] = AllocateMamePen(KRgbWhite);
	Machine->uifont->colortable[3] = AllocateMamePen(KRgbBlack);
}


__inline TUint8 MAKE_2BPP_BYTE(TUint8 c1, TUint8 c2, TUint8 c3, TUint8 c4)
{
	return (c4 << 6) | (c3 << 4) | (c2 << 2) | (c1);
}

__inline TUint8 MAKE_4BPP_BYTE(TUint8 c1, TUint8 c2)
{
	return (c2 << 4) | (c1);
}


void CMameScreen::DrawScaledScanLine8bppTo8bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge)
{
	TInt			pixelCount = 0;
	TUint8*	src = (TUint8*) aSrcLine;
	TUint8*	dst = (TUint8*) aDstLine;

	for (TInt w=0 ; w<aWidth ; w++)
	{
		if (pixelCount == iPixelSkipX)
		{
			pixelCount = 0;
		}
		else
		{
			if (aMerge)
			{
				*dst++ |= *src;
			}
			else
			{
				*dst++ = *src;
			}
			pixelCount++;
		}
		src++;
	}
}

void CMameScreen::DrawDoubleScaledScanLine8bppTo8bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge)
{
	TInt			pixelCount = 0;
	TUint8*	src = (TUint8*) aSrcLine;
	TUint8*	dst = (TUint8*) aDstLine;

	for (TInt w=0 ; w<aWidth ; w++)
	{
		if (pixelCount == iPixelSkipX)
		{
			pixelCount = 0;
			*(dst-1) |= *src;
			*(dst-2) |= *src;
		}
		else
		{
			if (aMerge)
			{
				*dst++ |= *src;
				*dst++ |= *src;
			}
			else
			{
				*dst++ = *src;
				*dst++ = *src;
			}
			pixelCount++;
		}
		src++;
	}
}

void CMameScreen::DrawScaledScanLine8bppTo2bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge)
{
	TInt			srcPixCount = 0;
	TInt			dstPixCount = 0;
	TInt			pixelCount = 0;
	TUint			dstPixels = 0;
	TUint			srcPixels = *aSrcLine++;
	for (TInt w=0 ; w<aWidth ; w++)
	{
		dstPixels |= (srcPixels & 3) << (dstPixCount << 1);
		srcPixCount++;
		srcPixels >>= 8;
		if (pixelCount != (iPixelSkipX -1))
			dstPixCount++;

		if (++pixelCount == iPixelSkipX)
			pixelCount=0;

		if (srcPixCount == 4)
		{
			srcPixels = *aSrcLine++;
			srcPixCount=0;
		}
		if (dstPixCount == 16)
		{
			if (aMerge)
			{
				*aDstLine |= dstPixels;
				aDstLine++;
			}
			else
				*aDstLine++ = dstPixels;
			dstPixCount=0;
			dstPixels=0;
		}
	}
}

void CMameScreen::DrawScaledScanLine8bppTo4bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge)
{
	TInt			srcPixCount = 0;
	TInt			dstPixCount = 0;
	TInt			pixelCount = 0;
	TUint			dstPixels = 0;
	TUint			srcPixels = *aSrcLine++;
	for (TInt w=0 ; w<aWidth ; w++)
	{
		dstPixels |= (srcPixels & 0xf) << (dstPixCount << 2);
		srcPixCount++;
		srcPixels >>= 8;
		if (pixelCount != (iPixelSkipX -1))
			dstPixCount++;

		if (++pixelCount == iPixelSkipX)
			pixelCount=0;

		if (srcPixCount == 4)
		{
			srcPixels = *aSrcLine++;
			srcPixCount=0;
		}
		if (dstPixCount == 8)
		{
			if (aMerge)
			{
				*aDstLine |= dstPixels;
				aDstLine++;
			}
			else
				*aDstLine++ = dstPixels;
			dstPixCount=0;
			dstPixels=0;
		}
	}
}


void CMameScreen::DrawScaledScanLine8bppTo12bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge)
{
	TInt			pixelCount = 0;
	TUint8*		src = (TUint8*) aSrcLine;
	TUint16*	dst = (TUint16*) aDstLine;

	for (TInt w=0 ; w<aWidth ; w++)
	{
		if (pixelCount == iPixelSkipX)
		{
			pixelCount = 0;
		}
		else
		{
			if (aMerge)
			{
				*dst++ |= iMamePalette[*src];
			}
			else
			{
				*dst++ = iMamePalette[*src];
			}
			pixelCount++;
		}
		src++;
	}
}

void CMameScreen::DirectDrawBitmap8bppTo4bpp(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdVisibleBitmap();

	TUint8* frameBufferLine = iBitmapStartAddress;
	TUint8* bitmapLine;

	TInt	height = (osdBitmap->height > iScreenSize.iHeight) ? iScreenSize.iHeight : osdBitmap->height;
	TInt	width = (osdBitmap->width > iScreenSize.iWidth) ? iScreenSize.iWidth : osdBitmap->width;
	for(TInt h=0 ; h<height ; h++)
	{
		TUint8* frameBuf = frameBufferLine;
		bitmapLine = osdBitmap->line[h];
		for (TInt w=0 ; w<width ; w+=2)
		{
			TUint8	c = MAKE_4BPP_BYTE(bitmapLine[0], bitmapLine[1]);
			*frameBuf++ = c;
			bitmapLine+=2;
		}
		frameBufferLine += iBytesPerScanLine;
	}
}



void CMameScreen::DirectDrawBitmap8bppTo2bpp(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdVisibleBitmap();

	TUint8* frameBufferLine = iBitmapStartAddress;
	TUint8* bitmapLine;

	TInt	height = (osdBitmap->height > iScreenSize.iHeight) ? iScreenSize.iHeight : osdBitmap->height;
	TInt	width = (osdBitmap->width > iScreenSize.iWidth) ? iScreenSize.iWidth : osdBitmap->width;
	for(TInt h=0 ; h<height ; h++)
	{
		TUint8* frameBuf = frameBufferLine;
		bitmapLine = osdBitmap->line[h];
		for (TInt w=0 ; w<width ; w+=4)
		{
			TUint8	c = MAKE_2BPP_BYTE(bitmapLine[0], bitmapLine[1], bitmapLine[2], bitmapLine[3]);
			*frameBuf++ = c;
			bitmapLine+=4;
		}
		frameBufferLine += iBytesPerScanLine;
	}
}

void CMameScreen::DirectDrawBitmap8bppTo12bpp(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdVisibleBitmap();

	TUint8* frameBufferLine = iBitmapStartAddress;
	TUint8* bitmapLine;

	TInt	height = (osdBitmap->height > iScreenSize.iHeight) ? iScreenSize.iHeight : osdBitmap->height;
	TInt	width = (osdBitmap->width > iScreenSize.iWidth) ? iScreenSize.iWidth : osdBitmap->width;
	for(TInt h=0 ; h<height ; h++)
	{
		TUint16* frameBuf = (TUint16* ) frameBufferLine;
		bitmapLine = osdBitmap->line[h];
		for (TInt w=0 ; w<width ; w++)
		{
			*frameBuf++ = (TUint16) iMamePalette[*bitmapLine++];
		}
		frameBufferLine += iBytesPerScanLine;
	}
}



void CMameScreen::DirectDrawBitmap8bppTo8bpp(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdVisibleBitmap();

	TUint8* frameBufferLine = iBitmapStartAddress;
	TUint8* bitmapLine;

	TInt	height = (osdBitmap->height > iScreenSize.iHeight) ? iScreenSize.iHeight : osdBitmap->height;
	TInt	width = (osdBitmap->width > iScreenSize.iWidth) ? iScreenSize.iWidth : osdBitmap->width;
	for(TInt h=0 ; h<height ; h++)
	{
		TUint8* frameBuf = frameBufferLine;
		bitmapLine = osdBitmap->line[h];
		Mem::Copy(frameBuf, bitmapLine, width);
		frameBufferLine += iBytesPerScanLine;
	}
}

void CMameScreen::ScaledDrawBitmap(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdVisibleBitmap();

	TUint8* frameBufLine = iBitmapStartAddress;

	TInt	lineCount = 0;
	
	TInt height = (iPixelSkipY>0) ? osdBitmap->height : ((osdBitmap->height > iScreenSize.iHeight) ? iScreenSize.iHeight : osdBitmap->height);
	TInt width = (iPixelSkipX>0) ? osdBitmap->width : ((osdBitmap->width > iScreenSize.iWidth) ? iScreenSize.iWidth : osdBitmap->width);

	TInt bytesPerBitmapLine = (iPixelSkipX>0) ? iWidth * CMameBitmap::GetBpp(iDisplayMode) / 8 : width * CMameBitmap::GetBpp(iDisplayMode) / 8;
	
	for (TInt h=0 ; h<height ; h++)
	{
		if (lineCount == iPixelSkipY)
		{
			(this->*iScaledScanLineDrawer)((TUint *) osdBitmap->line[h], (TUint*)iScanLine, width, ETrue);
			lineCount = 0;
		}
		else
		{
			(this->*iScaledScanLineDrawer)((TUint *) osdBitmap->line[h], (TUint*)iScanLine, width, EFalse);
			lineCount++;
		}
		if (lineCount != iPixelSkipY)
		{
			Mem::Copy(frameBufLine, iScanLine, bytesPerBitmapLine);
			frameBufLine += iBytesPerScanLine;
		}
	}
}

void CMameScreen::DoubleScaledDrawBitmap(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdVisibleBitmap();

	TUint8* frameBufLine = iBitmapStartAddress;

	TInt	lineCount = 0;
	
	TInt height = (iPixelSkipY>0) ? osdBitmap->height : ((osdBitmap->height > iScreenSize.iHeight) ? iScreenSize.iHeight : osdBitmap->height);
	TInt width = (iPixelSkipX>0) ? osdBitmap->width : ((osdBitmap->width > iScreenSize.iWidth) ? iScreenSize.iWidth : osdBitmap->width);

	TInt bytesPerBitmapLine = (iPixelSkipX>0) ? (iWidth * CMameBitmap::GetBpp(iDisplayMode) / 8) * 2 : (width * CMameBitmap::GetBpp(iDisplayMode) / 8) * 2;

	for (TInt h=0 ; h<height ; h++)
	{
		if (lineCount == iPixelSkipY)
		{
			(this->*iScaledScanLineDrawer)((TUint *) osdBitmap->line[h], (TUint*)iScanLine, width, ETrue);
			lineCount = 0;
		}
		else
		{
			(this->*iScaledScanLineDrawer)((TUint *) osdBitmap->line[h], (TUint*)iScanLine, width, EFalse);
			lineCount++;
		}
		if (lineCount != iPixelSkipY)
		{
			Mem::Copy(frameBufLine, iScanLine, iBytesPerBitmapLine);
			frameBufLine += iBytesPerScanLine;

			if ((iTotalGameFrames < 10) || (iGameFrameCount == iVideoFps))
			{
				Mem::Fill(frameBufLine, bytesPerBitmapLine, 0);
			}
			frameBufLine += iBytesPerScanLine;
		}
	}
}



void CMameScreen::DirectDrawDoubleBitmap8bppTo8bpp(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdVisibleBitmap();

	TUint8* frameBufferLine = iBitmapStartAddress;
	TUint8* bitmapLine;

	TInt	height = osdBitmap->height;
	TInt	width = osdBitmap->width;
	for(TInt h=0 ; h<height ; h++)
	{
		TUint8* frameBuf = frameBufferLine;
		bitmapLine = osdBitmap->line[h];
		for (TInt w=0 ; w<width ; w++)
		{
			*frameBuf++ = *bitmapLine;
			*frameBuf++ = *bitmapLine++;
		}
		frameBufferLine += iBytesPerScanLine;

		if ((iTotalGameFrames < 10) || (iGameFrameCount == iVideoFps))
		{
			Mem::Fill(frameBufferLine, width*2, 0);
		}
		frameBufferLine += iBytesPerScanLine;
	}
}

#if 0
void CMameScreen::DirectDrawDoubleBitmap8bppTo8bpp(CMameBitmap* aMameBitmap)
{
	const struct osd_bitmap*	osdBitmap = aMameBitmap->OsdBitmap();

	TUint8* frameBufferLine = iBitmapStartAddress;
	TUint8* bitmapLine;

	TInt	height = osdBitmap->height;
	TInt	width = osdBitmap->width;
	for(TInt h=0 ; h<height ; h++)
	{
		for (TInt i=0 ; i<2 ; i++)
		{
			TUint8* frameBuf = frameBufferLine;
			bitmapLine = osdBitmap->line[h];
			for (TInt w=0 ; w<width ; w++)
			{
				*frameBuf++ = *bitmapLine;
				*frameBuf++ = *bitmapLine++;
			}
			frameBufferLine += iBytesPerScanLine;
		}
	}
}
#endif

#ifdef __WINS__
CWinsMameScreen::~CWinsMameScreen()
{
	delete iEmulatedScreenBitmap;
}

CWinsMameScreen::CWinsMameScreen(const TSize& aScreenSize, TDisplayMode aDisplayMode)
{
	iScreenSize		= aScreenSize;
	iDisplayMode	= aDisplayMode;
	iHasFrameBuffer = ETrue;
}

void CWinsMameScreen::ConstructL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps)
{
	iEmulatedScreenBitmap = CMameBitmap::NewL(iScreenSize.iWidth, iScreenSize.iHeight, iDisplayMode);
	iFrameBuffer = (TUint8*) iEmulatedScreenBitmap->OsdVisibleBitmap()->line[0];
	iBytesPerScanLine = (iScreenSize.iWidth * CMameBitmap::GetBpp(iDisplayMode)) / 8;

	CMameScreen::ConstructL(aWidth, aHeight, aDepth, aOrientation, aFps);
};


#endif


/***********************************************
 *
 * MAME WINDOW
 *
 ***********************************************/

CMameWindow::CMameWindow()
{
}

CMameWindow::~CMameWindow()
{
	delete iWsScreen;
	delete iMameBitmapArray;	// we don't know the CMameBitmap objects
}

CMameWindow* CMameWindow::NewL()
{
	CMameWindow*	self = new(ELeave) CMameWindow;
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();
	return self;
}


void CMameWindow::ConstructL()
{
	TInt	error;

	iMameBitmapArray = new(ELeave) CMameBitmapArray(KMameBitmapArrayGranularity);

	error = iWsSession.Connect();
	User::LeaveIfError(error);
	iWsScreen=new(ELeave) CWsScreenDevice(iWsSession);
	User::LeaveIfError(iWsScreen->Construct());
	User::LeaveIfError(iWsScreen->CreateContext(iWindowGc));

	iWsWindowGroup=RWindowGroup(iWsSession);
	User::LeaveIfError(iWsWindowGroup.Construct((TUint32)this));
	iWsWindowGroup.SetOrdinalPosition(0);

	iWsWindow=RWindow(iWsSession);
	User::LeaveIfError(iWsWindow.Construct(iWsWindowGroup, (TUint32)this));
	iWsWindow.Activate();
	iWsWindow.SetSize(iWsScreen->SizeInPixels());
	iWsWindow.SetVisible(ETrue);

	StartWServEvents();
}

void CMameWindow::Destruct()
{
	if (iWsWindow.WsHandle())
		iWsWindow.Close();

	if (iWsWindowGroup.WsHandle())
		iWsWindowGroup.Close();

	delete iWindowGc;
	iWindowGc = NULL;

	delete iWsScreen;
	iWsScreen = NULL;

	if (iWsSession.WsHandle())
		iWsSession.Close();
}


void CMameWindow::CreateMameScreenL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps)
{
	TScreenInfoV01			screenInfo;
	TPckg<TScreenInfoV01>	sI(screenInfo);
	UserSvr::ScreenInfo(sI);

		/* 
		 * Use the scripts in \mame\src\epoc\group\wins to set the EPOC emulator 
		 * to s5mx or s7 appearance
		 */
#ifdef __WINS__
	iMameScreen = new CWinsMameScreen(iWsScreen->SizeInPixels(), iWsScreen->DisplayMode());
#else
	iMameScreen = new CMameScreen(screenInfo, iWsScreen->DisplayMode());
#endif

	iMameScreen->SetWantScaling(ETrue);
	iMameScreen->SetWantDoubleBitmap(iGameOptions.iGameSize == TGameOptions::EGameSizeFitScreen);

	iMameScreen->ConstructL(aWidth, aHeight, aDepth, aOrientation, aFps);
}

void CMameWindow::CreateDisplayL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps, const TGameOptions& aGameOptions)
{
	delete iMameScreen;
	iMameScreen = NULL;

	iGameOptions = aGameOptions;

	CreateMameScreenL(aWidth, aHeight, aDepth, aOrientation, aFps);
	SetupOnScreenControls();
	DrawOnScreenControlsNow();
}


CMameBitmap* CMameWindow::AllocateBitmapL(TInt aWidth, TInt aHeight, TInt aDepth)
{
	CMameBitmap*	mameBitmap;
	
	mameBitmap = CMameBitmap::NewL(aWidth, aHeight, CMameBitmap::GetDisplayMode(aDepth));
	iMameBitmapArray->Append(mameBitmap);

	return mameBitmap;
}



void CMameWindow::SetVisbibleAreaL(const TRect aRect)
{
	for (TInt i=0 ; i<iMameBitmapArray->Count() ; i++)
	{
		(*iMameBitmapArray)[i]->SetVisibleAreaL(aRect);
	}
}

void CMameWindow::MarkInvalid(TInt aXmin, TInt aYmin, TInt aXmax, TInt aYmax)
{
	TSize	size(iWsWindow.Size());
	TRect	rect(aXmin, aYmin, (aXmax == -1) ? size.iWidth : aXmax, (aYmax == -1) ? size.iHeight : aYmax);
}


TInt CMameWindow::SkipThisFrame()
{
	return iMameScreen->SkipThisFrame();
}

void CMameWindow::DrawBitmap(CMameBitmap* aMameBitmap)
{
	if (!iIsPaused)
	{
		if (input_ui_pressed(IPT_UI_SHOW_FPS))
		{
			iShowFps = iShowFps ? (TBool) EFalse : (TBool) ETrue;
		}


		TInt count = iMameScreen->DoubleFrame(iShowFps, aMameBitmap->OsdVisibleBitmap());
		count = 1;
		while(count--)
		{
			if (iMameScreen->HasFrameBuffer())
#ifdef __WINS__
				EmulateTargetDrawBitmap(aMameBitmap);
#else
				iMameScreen->DrawBitmap(aMameBitmap);
#endif
			else
				WsBlitBitmap(aMameBitmap);
		}
	}
}

#ifdef __WINS__
void CMameWindow::EmulateTargetDrawBitmap(CMameBitmap* aMameBitmap)
{
	iWindowGc->Activate(iWsWindow);

	TRect  rect = TRect(iMameScreen->Position(),  iMameScreen->GameBitmapSize());
	iWsWindow.Invalidate(rect);
	iWsWindow.BeginRedraw(rect);

	iMameScreen->DrawBitmap(aMameBitmap);
	iWindowGc->BitBlt(iMameScreen->Position(), ((CWinsMameScreen*) iMameScreen)->iEmulatedScreenBitmap->FbsBitmap(), rect);
	DrawOnScreenControls();

	iWsWindow.EndRedraw();
	iWindowGc->Deactivate();
	iWsSession.Flush();
}
#endif


void CMameWindow::WsBlitBitmap(CMameBitmap* aMameBitmap)
{
	iWindowGc->Activate(iWsWindow);

	TRect  rect = TRect(iWsWindow.Size());
	iWsWindow.Invalidate(rect);
	iWsWindow.BeginRedraw(rect);

	if (aMameBitmap)
	{
		iWindowGc->BitBlt(iMameScreen->Position(), aMameBitmap->FbsBitmap());
	}

	iWsWindow.EndRedraw();
	iWindowGc->Deactivate();
	iWsSession.Flush();
}




void CMameWindow::AllocateColor(TUint totalcolors, const UINT8 *palette, UINT16 *pens, TInt modifiable, const UINT8 *debug_palette,UINT16 *debug_pens)
{
	iMameScreen->AllocateColors(totalcolors, palette, pens, modifiable, debug_palette,debug_pens);
}



void CMameWindow::CloseDisplay()
{
	delete iMameScreen;
	iMameScreen = 0;

	CancelWServEvents();

	Destruct();
}

TInt TranslateKeyEvent(TInt aKeyCode)
{
	if (options.rol)
	{
		if ((aKeyCode>= EKeyLeftArrow) && (aKeyCode <= EKeyDownArrow))
		{
			const TInt	mappedKeys[] = {EKeyUpArrow, EKeyDownArrow, EKeyRightArrow, EKeyLeftArrow};
			return mappedKeys[aKeyCode - EKeyLeftArrow];
		}
	}
	else if (options.ror)
	{
		if ((aKeyCode>= EKeyLeftArrow) && (aKeyCode <= EKeyDownArrow))
		{
			const TInt	mappedKeys[] = {EKeyDownArrow, EKeyUpArrow, EKeyLeftArrow, EKeyRightArrow};
			return mappedKeys[aKeyCode - EKeyLeftArrow];
		}
	}
	return aKeyCode;
}

TInt CMameWindow::GetOnScreenKey(const TPoint aPosition)
{
	for (TInt i=0 ; i<MAX_ONSCREENCONTROLS ; i++)
	{
		if (iOnScreenControlList[i].iRect.Contains(aPosition))
			return iOnScreenControlList[i].iKey;
	}
	return -1;
}

void CMameWindow::HandlePointerEventL(const TPointerEvent& aPointerEvent)
{
	if (aPointerEvent.iType==TPointerEvent::EButton1Down)
	{
		iLastOnScreenKey = GetOnScreenKey(aPointerEvent.iPosition);
	}
	else if (aPointerEvent.iType==TPointerEvent::EButton1Up)
	{
		iLastOnScreenKey = -1;
	}
}


void CMameWindow::KeyDownEvent(TInt aScanCode)
{
	if (iKeyPressLevel < MAX_KEYPRESS)
	{
		iKeyPress[iKeyPressLevel].iScanCode = aScanCode;
		iKeyPressLevel++;
	}
}

void CMameWindow::KeyPressEvent(TInt aKeyCode)
{
	iKeyPress[iKeyPressLevel-1].iKeyCode = aKeyCode;
}

void CMameWindow::KeyUpEvent(TInt aScanCode)
{
	for (TInt i=iKeyPressLevel-1 ; i>=0 ; i--)
	{
		if (iKeyPress[i].iScanCode == aScanCode)
		{
			iKeyPress[i].iScanCode = iKeyPress[i].iKeyCode = 0;
			break;
		}
	}

	while(iKeyPressLevel>0 && (iKeyPress[iKeyPressLevel-1].iKeyCode == 0))
		iKeyPressLevel--;
}

TInt CMameWindow::CurrentKey()
{
	if (iKeyPressLevel == 0)
		return 0;
	return iKeyPress[iKeyPressLevel-1].iKeyCode;
}


void CMameWindow::HandleWsEvent(const TWsEvent& aWsEvent)
{
	DPRINT((HERE("CMameWindow::HandleWsEvent, type=%d"), aWsEvent.Type()));

	if (aWsEvent.Type() == EEventKeyDown)
	{
		KeyDownEvent(aWsEvent.Key()->iScanCode);
	} 
	else if (aWsEvent.Type() == EEventKey)
	{
		KeyPressEvent(TranslateKeyEvent(aWsEvent.Key()->iCode));
		if (aWsEvent.Key()->iCode == '+')
		{
			sManualDiffFrames++;
		}
		else if (aWsEvent.Key()->iCode == '-')
		{
			if (sManualDiffFrames > 0)
				sManualDiffFrames--;
		}
	} 
	else if (aWsEvent.Type() == EEventKeyUp)
	{
		KeyUpEvent(aWsEvent.Key()->iScanCode);
	} 
	else if (aWsEvent.Type() == EEventPointer)
	{
		HandlePointerEventL(*aWsEvent.Pointer());
	}
	else if(aWsEvent.Type() == EEventFocusLost)
	{
		if (!iIsPaused)
		{
			iLastOnScreenKey = 'p';		// simulate a pause
		}
	}
	else if(aWsEvent.Type() == EEventFocusGained)
	{
		if (iIsPaused)
			WsBlitBitmap((CMameBitmap*)Machine->scrbitmap->_private);		
	}
}


const KSmallAxis= 50;
const KLongAxis	= 100;

void CMameWindow::SetupOnScreenControls()
{
	TInt	heightDiv2 = iMameScreen->ScreenSize().iHeight / 2;
	TInt	width = iMameScreen->ScreenSize().iWidth;

	const TRect	horElipse(0, 0, KLongAxis, KSmallAxis);
	const TRect	verElipse(0, 0, KSmallAxis, KLongAxis);

	switch (iMameScreen->Orientation())
	{
		case EOrientationRotRight:
			iOnScreenControlList[0].iKey	= EKeyLeftArrow;
			iOnScreenControlList[0].iRect	= verElipse;
			iOnScreenControlList[0].iRect.Move(KLongAxis, heightDiv2 - (KSmallAxis / 2) - KLongAxis);

			iOnScreenControlList[1].iKey	= EKeyRightArrow;
			iOnScreenControlList[1].iRect	= verElipse;
			iOnScreenControlList[1].iRect.Move(KLongAxis, heightDiv2 + (KSmallAxis / 2));

			iOnScreenControlList[2].iKey	= EKeyUpArrow;
			iOnScreenControlList[2].iRect	= horElipse;
			iOnScreenControlList[2].iRect.Move(KLongAxis + KSmallAxis, heightDiv2 - (KSmallAxis / 2));

			iOnScreenControlList[3].iKey	= EKeyDownArrow;
			iOnScreenControlList[3].iRect	= horElipse;
			iOnScreenControlList[3].iRect.Move(0, heightDiv2 - (KSmallAxis / 2));
			break;

		case EOrientationRotLeft:
			iOnScreenControlList[0].iKey	= EKeyRightArrow;
			iOnScreenControlList[0].iRect	= verElipse;
			iOnScreenControlList[0].iRect.Move(width - KSmallAxis - KLongAxis, heightDiv2 - (KSmallAxis / 2) - KLongAxis);

			iOnScreenControlList[1].iKey	= EKeyLeftArrow;
			iOnScreenControlList[1].iRect	= verElipse;
			iOnScreenControlList[1].iRect.Move(width - KSmallAxis - KLongAxis, heightDiv2 + (KSmallAxis / 2));

			iOnScreenControlList[2].iKey	= EKeyUpArrow;
			iOnScreenControlList[2].iRect	= horElipse;
			iOnScreenControlList[2].iRect.Move(width - KLongAxis - (KLongAxis + KSmallAxis), heightDiv2 - (KSmallAxis / 2));

			iOnScreenControlList[3].iKey	= EKeyDownArrow;
			iOnScreenControlList[3].iRect	= horElipse;
			iOnScreenControlList[3].iRect.Move(width - KLongAxis, heightDiv2 - (KSmallAxis / 2));
			break;
	}
}

void CMameWindow::DrawOnScreenControlsNow()
{
	iWindowGc->Activate(iWsWindow);

	TRect  rect = TRect(iWsWindow.Size());
	iWsWindow.Invalidate(rect);
	iWsWindow.BeginRedraw(rect);

	DrawOnScreenControls();

	iWsWindow.EndRedraw();
	iWindowGc->Deactivate();
	iWsSession.Flush();
}

void CMameWindow::DrawOnScreenControls()
{
	iWindowGc->SetBrushStyle(CGraphicsContext::ESolidBrush);
	iWindowGc->SetBrushColor(KRgbBlack);
	for (TInt i=0 ; i<MAX_ONSCREENCONTROLS ; i++)
		if (iOnScreenControlList->iKey) iWindowGc->DrawEllipse(iOnScreenControlList[i].iRect);
}

void CMameWindow::HandleRedrawEvent(const TWsRedrawEvent& iRedrawEvent)
{
	DPRINT((HERE("CMameWindow::HandleRedrawEvent, handle=%d"), iRedrawEvent.Handle()));
	
	// Server initiated drawing not needed here
	WsBlitBitmap((CMameBitmap*)Machine->scrbitmap->_private);
}


void CMameWindow::StartWServEvents()
{
	iWsEventStatus = KRequestPending;
	iWsSession.EventReady(&iWsEventStatus);

	iRedrawEventStatus = KRequestPending;
	iWsSession.RedrawReady(&iRedrawEventStatus);
}

void CMameWindow::CancelWServEvents()
{
	if (iWsEventStatus != KRequestPending)
		iWsSession.EventReadyCancel();

	if (iRedrawEventStatus != KRequestPending)
		iWsSession.RedrawReadyCancel();
}


void CMameWindow::PollWServEvents()
{
	if (iWsEventStatus != KRequestPending)
	{
		iWsSession.GetEvent(iWsEvent);
		HandleWsEvent(iWsEvent);
		iWsEventStatus = KRequestPending;
		iWsSession.EventReady(&iWsEventStatus);
	}
}

TBool CMameWindow::IsKeyPressed(TInt aKey)
{
	PollWServEvents();

	TBool	isPressed = (aKey == CurrentKey()) || (aKey == iLastOnScreenKey);

	return isPressed;
}

void CMameWindow::Pause(TBool aPause)
{
	iIsPaused = aPause;
}



