/*******************************************************************
 *
 *	File:		OsdEpoc.cpp
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

extern "C" {
#include "osdepend.h"
#include "driver.h"

int osd_mark_vector_dirty(int x, int y);
}


#undef NULL

#include <e32std.h>
#include <f32file.h>

#include <stdio.h>
#include <stdarg.h>

#include "emame.h"
#include "video.h"
#include "audio.h"



/**********************************
 *
 * Global variables
 *
 **********************************/

extern "C" {
TBool				gSoundEnabled = EFalse;		// used by sndintrf
}



/**********************************
 *
 * Static variables
 *
 **********************************/

static CMameWindow*	sMameWindow = NULL;
static CMameAudio*	sMameAudio = NULL;
static TInt			sSafetyArea = 0;



int osd_init(void)
{
	DPRINT((HERE("osd_init")));
	return 0;
}

void osd_exit(void)
{
	DEBUGGER();
}

/* VERY IMPORTANT: the function must allocate also a "safety area" 8 pixels wide all */
/* around the bitmap. This is required because, for performance reasons, some graphic */
/* routines don't clip at boundaries of the bitmap. */
struct osd_bitmap *osd_alloc_bitmap(int width, int height, int depth)
{
	CMameBitmap*	mameBitmap = NULL;

	DPRINT((HERE("osd_alloc_bitmap: width=%d, height=%d, depth=%d"), width, height, depth));
	ASSERT(depth == 8);	//  for now we only support 8 bpp
	ASSERT(sMameWindow);

	TRAPD(err, mameBitmap = sMameWindow->AllocateBitmapL(width + sSafetyArea, height + sSafetyArea, depth));
	if (err)
	{
		logerror("osd_alloc_bitmap failed: width=%d, height=%d, depth=%d", width, height, depth);
	}

	return mameBitmap ? mameBitmap->OsdBitmap() : NULL;
}

void osd_clearbitmap(struct osd_bitmap *bitmap)
{
	DPRINT((HERE("osd_clearbitmap")));

	int i;

	for (i = 0;i < bitmap->height;i++)
	{
		if (bitmap->depth == 16)
			memset(bitmap->line[i],0,2*bitmap->width);
		else
			memset(bitmap->line[i], 0, bitmap->width);
	}


	if (bitmap == Machine->scrbitmap)
	{
		osd_mark_dirty(0, 0, -1 , -1 , 1);
	}
}

void osd_free_bitmap(struct osd_bitmap *bitmap)
{
	DEBUGGER();

	if (bitmap)
		delete (CMameBitmap*) bitmap->_private;
}

/* Create a display screen, or window, large enough to accomodate a bitmap */
/* of the given dimensions. Attributes are the ones defined in driver.h. */
/* Return a osd_bitmap pointer or 0 in case of error. */
int osd_create_display(int width,int height,int depth,int fps,int attributes,int orientation)
{
	DPRINT((HERE("osd_create_display: width=%d, height=%d, depth=%d, fps=%d, attrib=%d, orien=%d"), width, height, depth, fps, attributes, orientation));

	sMameWindow = CMameWindow::NewL();
		
	TOrientation	windowOrientation = EOrientationNormal;
	if (options.rol)
	{
		sSafetyArea = 16;
		windowOrientation = EOrientationRotLeft;
	}
	else if (options.ror)
	{
		sSafetyArea = 16;
		windowOrientation = EOrientationRotRight;
	}
	else
	{
		sSafetyArea = 0;
	}

	TRAPD(err, sMameWindow->CreateDisplayL(width + sSafetyArea, height + sSafetyArea, depth, windowOrientation, fps, GameOptions()));

	return err;
}

void osd_set_visible_area(int min_x, int max_x, int min_y, int max_y)
{
	DPRINT((HERE("osd_set_visible_area: min_x=%d, max_x=%d, min_y=%d, max_y=%d"), min_x, max_x, min_y, max_y));

	TRAPD(err, sMameWindow->SetVisbibleAreaL(TRect(min_x, min_y, max_x, max_y)));

	set_ui_visarea(min_x, min_y, max_x, max_y);
}

void osd_close_display(void)
{
	DEBUGGER();

		/*
		 * Get rid of these bitmaps before closing the display
		 */
	if (Machine->scrbitmap)
	{
		bitmap_free(Machine->scrbitmap);
		Machine->scrbitmap = NULL;
	}
	if (Machine->debug_bitmap)
	{
		osd_free_bitmap(Machine->debug_bitmap);
		Machine->debug_bitmap = NULL;
	}

	if (sMameWindow)
	{
		sMameWindow->CloseDisplay();
		delete sMameWindow;
		sMameWindow = NULL;
	}
}

/* palette is an array of 'totalcolors' R,G,B triplets. The function returns */
/* in *pens the pen values corresponding to the requested colors. */
/* If 'totalcolors' is 32768, 'palette' is ignored and the *pens array is filled */
/* with pen values corresponding to a 5-5-5 15-bit palette */


int osd_allocate_colors(unsigned int totalcolors,
		const UINT8 *palette,UINT16 *pens,int modifiable,
		const UINT8 *debug_palette,UINT16 *debug_pens)
{
	sMameWindow->AllocateColor(totalcolors, palette, pens, modifiable, debug_palette, debug_pens);
	return 0;
}

void osd_modify_pen(int pen,unsigned char red, unsigned char green, unsigned char blue)
{
	DEBUGGER();
}

void osd_get_pen(int pen,unsigned char *red, unsigned char *green, unsigned char *blue)
{
	DEBUGGER();
}

void osd_mark_dirty(int xmin, int ymin, int xmax, int ymax, int ui)
{
	DPRINT((HERE("osd_mark_dirty: xmin=%d, ymin=%d, xmax=%d, ymax=%d, ui=%d"), xmin, ymin, xmax, ymax, ui));
//	RDebug::Print(HERE("osd_mark_dirty: xmin=%d, ymin=%d, xmax=%d, ymax=%d, ui=%d"), xmin, ymin, xmax, ymax, ui);
	if (sMameWindow)
		sMameWindow->MarkInvalid(xmin, ymin, xmax, ymax);
}


int osd_mark_vector_dirty(int x, int y)
{
	DEBUGGER();
	return 1;
}

int osd_skip_this_frame(void)
{
	int		skip = 0;
	DPRINT((HERE("osd_skip_this_frame")));
	if (sMameWindow)
		skip= sMameWindow->SkipThisFrame();

	return skip;
}

void osd_update_video_and_audio(
		struct osd_bitmap *game_bitmap, struct osd_bitmap *debug_bitmap, int leds_status)
{
	DPRINT((HERE("osd_update_video_and_audio: led =%d"), leds_status));

	CMameBitmap*	mameBitmap = NULL;

	if (game_bitmap)
		mameBitmap = (CMameBitmap*) game_bitmap->_private;

	if (sMameWindow && mameBitmap)
		sMameWindow->DrawBitmap(mameBitmap);

	if (sMameAudio)
		sMameAudio->SoundUpdate();
}

void osd_debugger_focus(int debugger_has_focus)
{
	DEBUGGER();
}



void osd_set_gamma(float _gamma)
{
	DEBUGGER();
}

float osd_get_gamma(void)
{
	DEBUGGER();
	return 0.0;
}

void osd_set_brightness(int brightness)
{
	DEBUGGER();
}

int osd_get_brightness(void)
{
	DEBUGGER();
	return -1;
}

void osd_save_snapshot(struct osd_bitmap *bitmap)
{
	DEBUGGER();
}

/*************************
 *
 * Sound
 *
 *************************/

int osd_start_audio_stream(int stereo)
{
	int		samples = 0;

	DPRINT((HERE("osd_start_audio_stream: stereo=%d"), stereo));

	gSoundEnabled = GameOptions().iSoundEnabled;

	TInt samplesPerFrame = Machine->sample_rate / Machine->drv->frames_per_second;
	if (gSoundEnabled)
	{
		TRAPD(err, sMameAudio = CMameAudio::NewL((stereo == 0) ? (TBool) EFalse : (TBool) ETrue, samplesPerFrame));
		if (err)
			gSoundEnabled = EFalse;
	}
	samples = sMameAudio ? sMameAudio->FirstNoOfSamples() : 0;

	DPRINT((HERE("osd_start_audio_stream: done err=%d, samples=%d"), err, samples));
	
	return samples;
}


int osd_update_audio_stream(INT16 *buffer)
{
	TInt	samples = 0;

	DPRINT((HERE("osd_update_audio_stream")));
	
	if (sMameAudio)
		samples = sMameAudio->ProcessSoundSamples(buffer);

	return samples;
}

void osd_stop_audio_stream(void)
{
	DEBUGGER();
	if (sMameAudio)
	{
		delete sMameAudio;
		sMameAudio = NULL;
	}

	gSoundEnabled = EFalse;
}


void osd_set_mastervolume(int attenuation)
{
	DEBUGGER();
}

int osd_get_mastervolume(void)
{
	DEBUGGER();
	return 0;
}

void osd_sound_enable(int enable)
{
	DPRINT((HERE("osd_sound_enable, enable=%d"), enable));
}

/* direct access to the Sound Blaster OPL chip */
void osd_opl_control(int chip,int reg)
{
	DEBUGGER();
}

void osd_opl_write(int chip,int data)
{
	DEBUGGER();
}


/***********************
 *
 * Keyboard
 *
 ***********************/

#define KEY_A	'a'
#define KEY_B	'b'
#define KEY_C	'c'
#define KEY_D	'd'
#define KEY_E	'e'
#define KEY_F	'f'
#define KEY_G	'g'
#define KEY_H	'h'
#define KEY_I	'i'
#define KEY_J	'j'
#define KEY_K	'k'
#define KEY_L	'l'
#define KEY_M	'm'
#define KEY_N	'n'
#define KEY_O	'o'
#define KEY_P	'p'
#define KEY_Q	'q'
#define KEY_R	'r'
#define KEY_S	's'
#define KEY_T	't'
#define KEY_U	'u'
#define KEY_V	'v'
#define KEY_W	'w'
#define KEY_X	'x'
#define KEY_Y	'y'
#define KEY_Z	'z'
#define KEY_0	'0'
#define KEY_1	'1'
#define KEY_2	'2'
#define KEY_3	'3'
#define KEY_4	'4'
#define KEY_5	'5'
#define KEY_6	'6'
#define KEY_7	'7'
#define KEY_8	'8'
#define KEY_9	'9'

static struct KeyboardInfo keylist[] =
{
	{ "A",			KEY_A,				KEYCODE_A },
	{ "B",			KEY_B,				KEYCODE_B },
	{ "C",			KEY_C,				KEYCODE_C },
	{ "D",			KEY_D,				KEYCODE_D },
	{ "E",			KEY_E,				KEYCODE_E },
	{ "F",			KEY_F,				KEYCODE_F },
	{ "G",			KEY_G,				KEYCODE_G },
	{ "H",			KEY_H,				KEYCODE_H },
	{ "I",			KEY_I,				KEYCODE_I },
	{ "J",			KEY_J,				KEYCODE_J },
	{ "K",			KEY_K,				KEYCODE_K },
	{ "L",			KEY_L,				KEYCODE_L },
	{ "M",			KEY_M,				KEYCODE_M },
	{ "N",			KEY_N,				KEYCODE_N },
	{ "O",			KEY_O,				KEYCODE_O },
	{ "P",			KEY_P,				KEYCODE_P },
	{ "Q",			KEY_Q,				KEYCODE_Q },
	{ "R",			KEY_R,				KEYCODE_R },
	{ "S",			KEY_S,				KEYCODE_S },
	{ "T",			KEY_T,				KEYCODE_T },
	{ "U",			KEY_U,				KEYCODE_U },
	{ "V",			KEY_V,				KEYCODE_V },
	{ "W",			KEY_W,				KEYCODE_W },
	{ "X",			KEY_X,				KEYCODE_X },
	{ "Y",			KEY_Y,				KEYCODE_Y },
	{ "Z",			KEY_Z,				KEYCODE_Z },
	{ "0",			KEY_0,				KEYCODE_0 },
	{ "1",			KEY_1,				KEYCODE_1 },
	{ "2",			KEY_2,				KEYCODE_2 },
	{ "3",			KEY_3,				KEYCODE_3 },
	{ "4",			KEY_4,				KEYCODE_4 },
	{ "5",			KEY_5,				KEYCODE_5 },
	{ "6",			KEY_6,				KEYCODE_6 },
	{ "7",			KEY_7,				KEYCODE_7 },
	{ "8",			KEY_8,				KEYCODE_8 },
	{ "9",			KEY_9,				KEYCODE_9 },
	{ "Esc",		EKeyEscape,			KEYCODE_ESC },
	{ "Enter",		EKeyEnter,			KEYCODE_ENTER },
	{ "Up",			EKeyUpArrow,		KEYCODE_UP },
	{ "Down",		EKeyDownArrow,		KEYCODE_DOWN },
	{ "Left",		EKeyLeftArrow,		KEYCODE_LEFT },
	{ "Right",		EKeyRightArrow,		KEYCODE_RIGHT },
	{ 0, 0, 0 }	/* end of table */
};


const struct KeyboardInfo *osd_get_key_list(void)
{
	return keylist;
}

/*
  tell whether the specified key is pressed or not. keycode is the OS dependant
  code specified in the list returned by osd_get_key_list().
*/
int osd_is_key_pressed(int keycode)
{
	DPRINT((HERE("osd_is_key_pressed: keycode=%d"), keycode));

	int ret = sMameWindow ? sMameWindow->IsKeyPressed(keycode) : 0;

	if (keycode == EKeyEscape)
		if (GlobalData().iForcedExitFromFrontEnd)
			ret = 1;

	DPRINT((HERE("osd_is_key_pressed: ret=%d"), ret));

	return ret;
}

/*
  Return the Unicode value of the most recently pressed key. This
  function is used only by text-entry routines in the user interface and should
  not be used by drivers. The value returned is in the range of the first 256
  bytes of Unicode, e.g. ISO-8859-1. A return value of 0 indicates no key down.

  Set flush to 1 to clear the buffer before entering text. This will avoid
  having prior UI and game keys leak into the text entry.
*/
int osd_readkey_unicode(int flush)
{
	DEBUGGER();
	return 0;
}

static struct JoystickInfo joylist[1] =
{
	{ 0, 0, 0 }	/* end of table */
};

const struct JoystickInfo *osd_get_joy_list(void)
{
	return joylist;
}

/*
  tell whether the specified joystick direction/button is pressed or not.
  joycode is the OS dependant code specified in the list returned by
  osd_get_joy_list().
*/
int osd_joy_pressed(int joycode)
{
	DEBUGGER();
	return -1;
}

int osd_is_joy_pressed(int joycode)
{
	DEBUGGER();
	return -1;
}

/* Joystick calibration routines BW 19981216 */
/* Do we need to calibrate the joystick at all? */
int osd_joystick_needs_calibration (void)
{
	DPRINT((HERE("osd_joystick_needs_calibration")));
	return 0;
}

/* Preprocessing for joystick calibration. Returns 0 on success */
void osd_joystick_start_calibration (void)
{
	DEBUGGER();
}

/* Prepare the next calibration step. Return a description of this step. */
/* (e.g. "move to upper left") */
char *osd_joystick_calibrate_next (void)
{
	DEBUGGER();
	return (char *)NULL;
}
/* Get the actual joystick calibration data for the current position */
void osd_joystick_calibrate (void)
{
	DEBUGGER();
}

/* Postprocessing (e.g. saving joystick data to config) */
void osd_joystick_end_calibration (void)
{
	DEBUGGER();
}



void osd_trak_read(int player,int *deltax,int *deltay)
{
	DPRINT((HERE("osd_trak_read")));
	*deltax = *deltay = 0;
}

/* return values in the range -128 .. 128 (yes, 128, not 127) */
void osd_analogjoy_read(int player,int *analog_x, int *analog_y)
{
	DPRINT((HERE("osd_analogjoy_read")));
	*analog_x = *analog_y = 0;
}

void osd_customize_inputport_defaults(struct ipd *defaults)
{
	DPRINT((HERE("osd_customize_inputport_defaults")));
	while (defaults->type != IPT_END)
	{
		if (defaults->type == IPT_UI_SHOW_PROFILER) seq_set_1(&defaults->seq, KEYCODE_A);
		if (defaults->type == IPT_UI_SHOW_FPS) seq_set_1(&defaults->seq, KEYCODE_B);
		if (defaults->type == IPT_UI_SNAPSHOT) seq_set_1(&defaults->seq, KEYCODE_S);
		if (defaults->type == IPT_BUTTON1) seq_set_1(&defaults->seq, KEYCODE_Q);

		defaults++;
	}
}

/* gamename holds the driver name, filename is only used for ROMs and    */
/* samples. If 'write' is not 0, the file is opened for write. Otherwise */
/* it is opened for read. */
void CLIB_DECL logerror(const char *text,...)
{
	va_list marker;

//	printf("ERROR: ");
//	va_start( marker, text);
//	vprintf(text, marker);
}


/* control keyboard leds or other indicators */
void osd_led_w(int led,int on)
{
	DEBUGGER();
}


/*
To start profiling a certain section, e.g. video:
osd_profiler(OSD_PROFILE_VIDEO);
to end profiling of the current section:
osd_profiler(OSD_PROFILE_END);
*/

void osd_profiler(int type)
{
	DEBUGGER();
}

void osd_pause(int paused)
{
	DPRINT((HERE("osd_pause, paused=%d"), paused));
	if (sMameWindow)
		sMameWindow->Pause((TBool) paused);
}

#ifdef MAME_NET
/* network */
int osd_net_init(void){return 0;}
int osd_net_send(int player, unsigned char buf[], int *size){return 0;}
int osd_net_recv(int player, unsigned char buf[], int *size){return 0;}
int osd_net_sync(void){return 0;}
int osd_net_input_sync(void){return 0;}
int osd_net_exit(void){return 0;}
int osd_net_add_player(void){return 0;}
int osd_net_remove_player(int player){return 0;}
int osd_net_game_init(void){return 0;}
int osd_net_game_exit(void){return 0;}
#endif /* MAME_NET */




int stricmp( const char *string1, const char *string2 )
{
	DEBUGGER();
	return 0;
}



unsigned int osd_cycles()
{
	return User::TickCount();
}

void my_logerror(char *s)
{
	RDebug::Print(HERE("%s"), s);
}
