/*******************************************************************
 *
 *	File:		Fileio.cpp
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

extern "C" {
#include "driver.h"
#include "unzip.h"
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>

extern unsigned int crc32 (unsigned int crc, const unsigned char *buf, unsigned int len);


/* Verbose outputs to error.log ? */
#define VERBOSE 	0

#if VERBOSE
#define LOG(x)	logerror x
#else
#define LOG(x)	/* x */
#endif

}		// extern "C"




#include <e32std.h>
#include <e32svr.h>

#include "emame.h"


#define EMAME_ROM_DIR		"D:\\EMAME\\ROMS"
#define EMAME_DATA_DIR		"D:\\EMAME\\DATA"
#define EMAME_HI_DIR		"D:\\EMAME\\HI"
#define EMAME_CFG_DIR		EMAME_DATA_DIR
#define EMAME_INP_DIR		EMAME_DATA_DIR
#define EMAME_STATE_DIR		EMAME_DATA_DIR
#define EMAME_ART_DIR		EMAME_DATA_DIR
#define EMAME_MEMCRD_DIR	EMAME_DATA_DIR
#define EMAME_SCRNSHOT_DIR	EMAME_DATA_DIR
#define EMAME_CHEAT_DIR		EMAME_DATA_DIR


/*****************************************************
 *
 * EMAME File I/O interface 
 *
 *****************************************************/

class MMameFileInterface
{
public:
	virtual TInt Read(TAny* aBuffer, TInt aLength) = 0;
	virtual TInt Write(const TAny* aBuffer, TInt aLength) = 0;
	virtual TInt Seek(TInt aOffset, TInt aWhence) = 0;
	virtual TInt Close() = 0;
	virtual TInt Size() = 0;
	virtual TInt Tell() = 0;
	virtual TInt Getc() = 0;
	virtual TInt Ungetc(TInt aChar) = 0;
	virtual char* Gets(char *aString, TInt aCount) = 0;
	virtual TInt End() = 0;
	virtual TUint Crc() = 0;
};

// force interface to be CBased.
class CMameFileBase : public CBase, public MMameFileInterface
{
};

/*****************************************************
 *
 * CMamePlainFile
 *
 *****************************************************/


class CMamePlainFile : public CMameFileBase
{
public:
	~CMamePlainFile();
	static CMamePlainFile*	New(const char* aName, const char* aMode);

	virtual TInt Read(TAny* aBuffer, TInt aLength) {return fread(aBuffer, 1, aLength, iFile);}
	virtual TInt Write(const TAny* aBuffer, TInt aLength) {return fwrite(aBuffer, 1, aLength, iFile);}
	virtual TInt Seek(TInt aOffset, TInt aWhence) {return fseek(iFile, aOffset, aWhence);}
	virtual TInt Close();
	virtual TInt Size();
	virtual TInt Tell() {return ftell(iFile);}
	virtual TInt Getc() {return fgetc(iFile);}
	virtual TInt Ungetc(TInt aChar){return ungetc(aChar, iFile);}
	virtual char*  Gets(char *aString, TInt aCount) {return fgets(aString, aCount, iFile);}
	virtual TInt End(){return feof(iFile);}
	virtual TUint Crc();

protected:
	TBool Open(const char* aName, const char* aMode);
	FILE*	iFile;
};



CMamePlainFile*	CMamePlainFile::New(const char* aName, const char* aMode)
{
	CMamePlainFile*	self = new CMamePlainFile;
	if (self)
	{
		if (!self->Open(aName, aMode))
		{
			delete self;
			self= NULL;
		}
	}
	return self;
}

CMamePlainFile::~CMamePlainFile()
{
	Close();
}

TBool CMamePlainFile::Open(const char* aName, const char* aMode)
{
	iFile = fopen(aName, aMode);
	return (iFile != NULL);
}



TInt CMamePlainFile::Close()
{
	TInt rval = 0;
	if (iFile)
	{
		rval = fclose(iFile);
		iFile = NULL;
	}
	return rval;
}

TInt CMamePlainFile::Size()
{
	int size, offs;
	offs = ftell(iFile);
	fseek(iFile, 0, SEEK_END );
	size = ftell(iFile);
	fseek(iFile, offs, SEEK_SET );
	return size;
}

TUint CMamePlainFile::Crc()
{
	int				length, fpos;
	unsigned char*	data;
	unsigned int	crc = 0;

	length	= Size();
	fpos	= Tell();

	data = (unsigned char *) malloc (length);
	if(data)
	{
		if(Seek(0L, SEEK_SET) == 0)
		{
			if (Read(data, length) == length)
			{
				crc = crc32 (0L, data, length);
			}
			Seek(fpos, 0);
		}
		free (data);
	}

	return crc;
}


/*****************************************************
 *
 * CMameRamFile
 *
 *****************************************************/

class CMameRamFile : public CMameFileBase
{
public:
	~CMameRamFile();
	static CMameRamFile* New(const char* aZipFileName, const char* aFileName);
	virtual TInt Read(TAny* aBuffer, TInt aLength);
	virtual TInt Write(const TAny* aBuffer, TInt aLength) {return -1;}	// no can do
	virtual TInt Seek(TInt aOffset, TInt aWhence);
	virtual TInt Close();
	virtual TInt Size() {return iLength;}
	virtual TInt Tell() {return iOffset;}
	virtual TInt Getc() {return -1;}	// no can do
	virtual TInt Ungetc(TInt aChar) {return -1;}	// no can do
	virtual char* Gets(char *aString, TInt aCount) {return NULL;}	// no can do
	virtual TInt End(){return iLength == iOffset;}
	virtual TUint Crc() {return iCrc;}

protected:
	TBool OpenZipFile(const char* aZipFileName, const char* aFileName);

	unsigned char*	iData;
	unsigned int	iOffset;
	unsigned int	iLength;
	unsigned int	iCrc;
};


CMameRamFile* CMameRamFile::New(const char* aZipFileName, const char* aFileName)
{
	CMameRamFile*	self = new CMameRamFile;
	if (self)
	{
		if (!self->OpenZipFile(aZipFileName, aFileName))
		{
			delete self;
			self= NULL;
		}
	}
	return self;
}


TBool CMameRamFile::OpenZipFile(const char* aZipFileName, const char* aFileName)
{
	TBool	success = EFalse;

	if(load_zipped_file(aZipFileName, aFileName, &iData, &iLength) == 0)
	{
		iCrc = crc32(0L, iData, iLength);
		success = ETrue;
	}

	return success;
}


CMameRamFile::~CMameRamFile()
{
	Close();
}

TInt CMameRamFile::Read(TAny* aBuffer, TInt aLength)
{
	ASSERT(iData);

	if (aLength + iOffset > iLength)
		aLength = iLength - iOffset;
	memcpy(aBuffer, iOffset + iData, aLength);
	iOffset += aLength;

	return aLength;
}



TInt CMameRamFile::Seek(TInt aOffset, TInt aWhence)
{
	int	rval = 0;
	switch (aWhence)
	{
		case SEEK_SET:
			iOffset = aOffset;
			break;
		case SEEK_CUR:
			iOffset += aOffset;
			break;
		case SEEK_END:
			iOffset = iLength + iOffset;
			break;
		default:
			rval = -1;
	}
	return rval;
}


TInt CMameRamFile::Close()
{
	if (iData)
	{
		User::Free(iData);
		iData = NULL;
	}
	return 0;
}


/*****************************************************
 *
 * CMameFileFactory
 *
 *****************************************************/

class CMameFileFactory
{
public:
	static CMameFileBase*	Open(const char *aGame, const char *aFilename, int aFiletype, int aWrite);

protected:
	static CMamePlainFile*	OpenPlainFile(const char *aGame, const char *aFilename, int aFiletype, int aWrite);
	static CMameRamFile*	OpenZippedFile(const char *aGame, const char *aFilename, int aFiletype, int aWrite);
};


CMameFileBase* CMameFileFactory::Open(const char *aGame, const char *aFilename, int aFiletype, int aWrite)
{
	CMameFileBase*	file = NULL;
	switch(aFiletype)
	{
		case OSD_FILETYPE_ROM:
		case OSD_FILETYPE_SAMPLE:
			file = OpenZippedFile(aGame, aFilename, aFiletype, aWrite);
			if (!file)
				file = OpenPlainFile(aGame, aFilename, aFiletype, aWrite);
			break;
		default:
			file = OpenPlainFile(aGame, aFilename, aFiletype, aWrite);
			break;
	}
			
	return file;
}


CMameRamFile* CMameFileFactory::OpenZippedFile(const char *aGame, const char *aFileName, int aFiletype, int aWrite)
{
	char				name[256] = {0};
	CMameRamFile*		file = NULL;

	sprintf(name, "%s\\%s.zip", EMAME_ROM_DIR, aGame);

		/*
		 * verify existence of zipfile before hand, otherwise the unzip routines spit
		 * out a lot of errors
		 */
	if (access(name, 0) == 0)
	{
		file = CMameRamFile::New(name, aFileName);
	}
	else
	{
		name[0] = 'C';
		if (access(name, 0) == 0)
		{
			file = CMameRamFile::New(name, aFileName);
		}
	}

	return file;
}


CMamePlainFile* CMameFileFactory::OpenPlainFile(const char *aGame, const char *aFileName, int aFiletype, int aWrite)
{
	char				name[256] = {0};
	CMamePlainFile*		file = NULL;
	int					binary = 1;

	switch(aFiletype)
	{
		case OSD_FILETYPE_ROM:
		case OSD_FILETYPE_SAMPLE:
			sprintf(name, "%s\\%s\\%s", EMAME_ROM_DIR, aGame, aFileName);
			break;

		case OSD_FILETYPE_CONFIG:
			sprintf (name, "%s\\%s.cfg", EMAME_CFG_DIR, aFileName);
			break;

		case OSD_FILETYPE_INPUTLOG:
			sprintf (name, "%s\\%s.inp", EMAME_INP_DIR, aFileName);
			break;

		case OSD_FILETYPE_STATE:
			sprintf (name, "%s\\%s.sta", EMAME_STATE_DIR, aFileName);
			break;

		case OSD_FILETYPE_ARTWORK:
			sprintf (name, "%s\\%s", EMAME_ART_DIR, aFileName);
			break;

		case OSD_FILETYPE_MEMCARD:
			sprintf (name, "%s\\%s", EMAME_MEMCRD_DIR, aFileName);
			break;

		case OSD_FILETYPE_SCREENSHOT:
			sprintf (name, "%s\\%s.png", EMAME_SCRNSHOT_DIR, aFileName);
			break;

		case OSD_FILETYPE_CHEAT:
			sprintf (name, "%s\\%s", EMAME_CHEAT_DIR, aFileName);
			binary = 0;
			break;

			// not supported (yet)
		case OSD_FILETYPE_HIGHSCORE_DB:
			sprintf (name, "%s\\%s", EMAME_HI_DIR, aFileName);
			binary = 0;
			RDebug::Print(HERE("OSD_FILETYPE_HIGHSCORE_DB: game=%s, filename=%s, filetype=%d, _write=%d"), aGame ? aGame : "", aFileName ? aFileName : "", aFiletype, aWrite);
			break;

		case OSD_FILETYPE_HIGHSCORE:
			sprintf (name, "%s\\%s", EMAME_HI_DIR, aFileName);
			binary = 0;
			RDebug::Print(HERE("OSD_FILETYPE_HIGHSCORE: game=%s, filename=%s, filetype=%d, _write=%d"), aGame ? aGame : "", aFileName ? aFileName : "", aFiletype, aWrite);
			break;

		case OSD_FILETYPE_HISTORY:
		case OSD_FILETYPE_LANGUAGE:
		case OSD_FILETYPE_NVRAM:
			break;

		default:
			logerror("CMameFileFactory::OpenPlainFile(): unknown filetype %02x\n", aFiletype);
	}

	if (strlen(name))
	{
			/*
			 * We try to open by default on D, failing that we try on C
			 */
		file = binary ? CMamePlainFile::New(name, aWrite ? "wb" : "rb") : CMamePlainFile::New(name, aWrite ? "w" : "r");
		if (!file)
		{
			name[0] = 'C';
			file = binary ? CMamePlainFile::New(name, aWrite ? "wb" : "rb") : CMamePlainFile::New(name, aWrite ? "w" : "r");
		}
	}

	return file;
}


/********************************************
 *
 * OSD file I/O functions
 *
 ********************************************/



/*
 * file handling routines
 *
 * filename holds the driver name, filename is only used for ROMs and samples.
 * if 'write' is not 0, the file is opened for write. Otherwise it is opened
 * for read.
 */
int osd_faccess (const char *newfilename, int filetype)
{
	TAny*		file;
	int			found = 0;
	
	DPRINT((HERE("osd_faccess: newfilename=%s, filetype=%d"), newfilename, filetype));

	file = osd_fopen (newfilename, newfilename, filetype, 0);
	if (file)
	{
		osd_fclose(file);
		found = 1;
	}
	DPRINT((HERE("osd_faccess: found=%d"), found));

	return found;
}



void *osd_fopen(const char *game, const char *filename, int filetype, int _write)
{
	CMameFileBase*		file = NULL;

	DPRINT((HERE("osd_fopen: game=%s, filename=%s, filetype=%d, _write=%d"), game ? game : "", filename ? filename : "", filetype, _write));
	RDebug::Print(HERE("osd_fopen: game=%s, filename=%s, filetype=%d, _write=%d"), game ? game : "", filename ? filename : "", filetype, _write);

	file = CMameFileFactory::Open(game, filename, filetype, _write);

	DPRINT((HERE("osd_fopen: file=0x%x"), file));
	RDebug::Print(HERE("osd_fopen: file=0x%x"), file);

	return file;
}


int osd_fread (void *file, void *buffer, int length)
{
	int		size;

	DPRINT((HERE("osd_fread : file=0x%x, length=%d"), file, length));

	size = 	((CMameFileBase*) file)->Read(buffer, length);

	DPRINT((HERE("osd_fread : size=%d"), size));

	return size;
}

int osd_fread_swap (void *file, void *buffer, int length)
{
	int i;
	unsigned char *buf;
	unsigned char temp;
	int res;

	DPRINT((HERE("osd_fread_swap: file=0x%x, length=%d"), file, length));

	res = ((CMameFileBase*) file)->Read(buffer, length);

	buf = (unsigned char*) buffer;
	for( i = 0; i < length; i += 2 )
	{
		temp = buf[i];
		buf[i] = buf[i + 1];
		buf[i + 1] = temp;
	}

	DPRINT((HERE("osd_fread_swap: res=%d"), res));

	return res;
}


int osd_fwrite (void *file, const void *buffer, int length)
{
	int		size;
	
	DPRINT((HERE("osd_fwrite : file=0x%x, length=%d"), file, length));

	size = ((CMameFileBase*) file)->Write(buffer, length);
	
	DPRINT((HERE("osd_fwrite : size=%d"), size));

	return size;
}


int osd_fwrite_swap (void *file, const void *buffer, int length)
{
	int i;
	unsigned char *buf;
	unsigned char temp;
	int res;

	DPRINT((HERE("osd_fwrite_swap: file=0x%x, length=%d"), file, length));

	buf = (unsigned char *) buffer;
	for( i = 0; i < length; i += 2 )
	{
		temp = buf[i];
		buf[i] = buf[i + 1];
		buf[i + 1] = temp;
	}

	res = ((CMameFileBase*) file)->Write(buffer, length);

	for( i = 0; i < length; i += 2 )
	{
		temp = buf[i];
		buf[i] = buf[i + 1];
		buf[i + 1] = temp;
	}

	DPRINT((HERE("osd_fwrite_swap: res=%d"), res));

	return res;
}

int osd_fread_scatter (void *file, void *buffer, int length, int increment)
{
	unsigned char *buf = (unsigned char *) buffer;
	unsigned char tempbuf[4096];
	int totread, r, i;

	DPRINT((HERE("osd_fread_scatter: file=0x%x, length=%d, increment=%d"), file, length, increment));

	totread = 0;
	while (length)
	{
		r = length;
		if( r > 4096 )
			r = 4096;
		r = ((CMameFileBase*) file)->Read(tempbuf, r);
		if( r == 0 )
			return totread;		   /* error */
		for( i = 0; i < r; i++ )
		{
			*buf = tempbuf[i];
			buf += increment;
		}
		totread += r;
		length -= r;
	}

	DPRINT((HERE("osd_fread_scatter: totread=%d"), totread));

	return totread;
}


int osd_fseek (void *file, int offset, int whence)
{
	int		ret;

	DPRINT((HERE("osd_fseek: file=0x%x, offset=%d, whence=%d"), file, offset, whence));

	ret = ((CMameFileBase*) file)->Seek(offset, whence);

	DPRINT((HERE("osd_fseek: ret=%d"), ret));

	return ret;
}


void osd_fclose (void *file)
{
	DPRINT((HERE("osd_fclose: file=0x%x"), file));

	((CMameFileBase*) file)->Close();
	delete ((CMameFileBase*) file);
}


int osd_fchecksum (const char *game, const char *filename, unsigned int *length, unsigned int *sum)
{

	DPRINT((HERE("osd_fchecksum: game=%s, filename=%s"), game, filename));

	DEBUGGER();

#if 0
	/* Support "-romdir" yuck. */
	if( alternate_name )
		filename = alternate_name;

	for( indx = 0; indx < rompathc && !found; ++indx )
	{
		const char *dir_name = rompathv[indx];

		if( !found )
		{
			sprintf (name, "%s/%s", dir_name, filename);
			if( cache_stat (name, &stat_buffer) == 0 && (stat_buffer.st_mode & S_IFDIR) )
			{
				sprintf (name, "%s/%s/%s", dir_name, filename, filename);
				if( checksum_file (name, 0, length, sum) == 0 )
                {
					found = 1;
                }
			}
		}

		if( !found )
		{
			/* try with a .zip extension */
			sprintf (name, "%s/%s.zip", dir_name, filename);
			if( cache_stat (name, &stat_buffer) == 0 )
			{
				if( checksum_zipped_file (name, filename, length, sum) == 0 )
				{
					LOG(("Using (osd_fchecksum) zip file for %s\n", filename));
					found = 1;
				}
			}
		}

		if( !found )
		{
			/* try with a .zif directory (if ZipFolders is installed) */
			sprintf (name, "%s/%s.zif", dir_name, filename);
			if( cache_stat (name, &stat_buffer) == 0 )
			{
				sprintf (name, "%s/%s.zif/%s", dir_name, filename, filename);
				if( checksum_file (name, 0, length, sum) == 0 )
				{
					found = 1;
				}
			}
		}
	}

	if( !found )
		return -1;
#endif
	return 0;
}

/* JB 980920 */
int osd_fsize (void *file)
{
	DPRINT((HERE("osd_fsize: file=0x%x"), file));

	int size = ((CMameFileBase*) file)->Size();

	DPRINT((HERE("osd_fsize: size=%d"), size));

	return size;
}

unsigned int osd_fcrc(void *file)
{
	int				length, fpos;
	unsigned char*	data;
	unsigned int	crc = 0;
	CMameFileBase* mameFile = (CMameFileBase*) file;

	DPRINT((HERE("osd_fcrc : file=0x%x"), file));

	crc = mameFile->Crc();

	DPRINT((HERE("osd_fcrc : crc=0x%x"), crc));

	return crc;
}


int osd_fgetc(void *file)
{
	int		c;

	DPRINT((HERE("osd_fgetc: file=0x%x"), file));

	c = ((CMameFileBase*) file)->Getc();

	DPRINT((HERE("osd_fgetc: c=%d"), c));

	return c;

}

int osd_ungetc(int c, void *file)
{
	int		ret;

	DPRINT((HERE("osd_ungetc: file=0x%x, c=%d"), file, c));

	ret = ((CMameFileBase*) file)->Ungetc(c);

	DPRINT((HERE("osd_ungetc: c=%d"), ret));
	
	return ret;
}

char *osd_fgets(char *s, int n, void *file)
{
	char*	ret;

	DPRINT((HERE("osd_fgets: file=0x%x, n=%d"), file, n));

	ret = ((CMameFileBase*) file)->Gets(s, n);

	DPRINT((HERE("osd_fgets: s=%s"), s));

	return ret;
}

int osd_feof(void *file)
{
	int		ret;

	DPRINT((HERE("osd_feof: file=0x%x"), file));

	ret = ((CMameFileBase*) file)->End();

	DPRINT((HERE("osd_feof: ret=%d"), ret));

	return ret;
}

int osd_ftell(void *file)
{
	int		ret;

	DPRINT((HERE("osd_ftel: file=0x%x"), file));

	ret = ((CMameFileBase*) file)->Tell();

	DPRINT((HERE("osd_ftel: ret=%d"), ret));

	return ret;
}


/* called while loading ROMs. It is called a last time with name == 0 to signal */
/* that the ROM loading process is finished. */
/* return non-zero to abort loading */
int osd_display_loading_rom_message (const char *name, int current, int total)
{
	DPRINT((HERE("osd_display_loading_rom_message : name=%s, current=%d, total=%d"), name ? name : "", current, total));
	if( name )
		fprintf (stdout, "loading %-12s\n", name);
	fflush (stdout);

	return 0;
}
