/*******************************************************************
 *
 *	File:		EMame.cpp
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

extern "C" {
#include "osdepend.h"
#include "mame.h"
#include "driver.h"
#include "platform.h"

#ifdef __WINS__
	/*
	 * From sndintrf.c
	 */
struct snd_interface
{
	unsigned sound_num;										/* ID */
	const char *name;										/* description */
	int (*chips_num)(const struct MachineSound *msound);	/* returns number of chips if applicable */
	int (*chips_clock)(const struct MachineSound *msound);	/* returns chips clock if applicable */
	int (*start)(const struct MachineSound *msound);		/* starts sound emulation */
	void (*stop)(void);										/* stops sound emulation */
	void (*update)(void);									/* updates emulation once per frame if necessary */
	void (*reset)(void);									/* resets sound emulation */
};
struct snd_interface sndintf[];
#endif

}

#undef NULL

#include <stdio.h>
#include <unistd.h>
#include <e32std.h>
#include <f32file.h>
#include <e32svr.h>
#include <estlib.h>
#include <eikdll.h>
#include <w32std.h>


#include "emame.h"
#include "globaldata.h"


static RChunk			sGlobalDataChunk;
static TGlobalData		sGlobalData;
static TInt				sContrast = -1;

/*
 * We need to get the options to 
 */
TGameOptions			sGameOptions;

const TGameOptions& GameOptions()
{
	return sGameOptions;
}

TGlobalData& GlobalData()
{
	return sGlobalData;
}

void SendMameCloseEventL()
{
	TWsEvent	wsEvent;
	wsEvent.SetType(EMameShuttingDown);

	TAutoClose<RWsSession> wsSession;

	User::LeaveIfError(wsSession.iObj.Connect());
	wsSession.PushL();
	User::LeaveIfError(wsSession.iObj.SendEventToWindowGroup(sGlobalData.iAppWindowGroupId, wsEvent));
	CleanupStack::PopAndDestroy();
}

void SendMameCloseEvent()
{
	TRAPD(err, SendMameCloseEventL());
	if (err)
		User::InfoPrint(_L("EMAME: Please terminate the thread STDOUT in the task list"));
}


void ShutdownMame()
{
	sGlobalDataChunk.Close();
}


void SetGamesOptions(const TGameOptions& aGameOptions)
{
	switch (aGameOptions.iOrientation)
	{
		case TGameOptions::EOrientationLeft:
			options.rol = 1;
			break;

		case TGameOptions::EOrientationRight:
			options.ror = 1;
			break;

		case TGameOptions::EOrientationUpsideDown:
			options.flipy = 1;
			break;

		case TGameOptions::EOrientationNone:
		default:
			break;
	}
	if (aGameOptions.iContrast == TGameOptions::EContrastSetToLow)
	{
		PLATFORM_DISPLAYCONTRAST(sContrast);
		PLATFORM_SETDISPLAYCONTRAST(1);
	}
	else
		sContrast = -1;

	sGameOptions = aGameOptions;		// and take a copy
}

void RestoreGlobalSettings()
{
	if (sContrast != -1)
	{
		PLATFORM_SETDISPLAYCONTRAST(sContrast);
		sContrast = -1;
	}
}


void SetDefaultGamesOptions()
{
#ifdef _DEBUG
    options.mame_debug		=1;
#else
    options.mame_debug		= 0;
#endif

	options.record			= 0;
	options.playback		= 0;
	options.cheat			= 0;
	options.gui_host		= 1;

	options.samplerate			= 8000;
	options.use_samples			= 0;
	options.use_emulated_ym3812 = 0;

	options.norotate		= 0;
	options.ror				= 0;
	options.rol				= 0;
	options.flipx			= 0;
	options.flipy			= 0;

	options.translucency	= 0;
	options.antialias		= 0;
	options.use_artwork		= 0;

	options.beam   = (int)(1 /*playing_game_options.beam*/ * 0x00010000);
	if (options.beam < 0x00010000)    /* beam MUST be at least 1 pixel */
		options.beam = 0x00010000;
	if (options.beam > 0x00100000)    /* set max to 16 pixels */
		options.beam = 0x00100000;

	options.flicker   = (int)(100 /*playing_game_options.flicker*/ * 2.55);
	if (options.flicker < 0)
		options.flicker = 0;
	if (options.flicker > 255)
		options.flicker = 255;
}

#define CPU_TYPE(type) 	(type & ~CPU_FLAGS_MASK)
#define CPU_AUDIO(type)	(type & CPU_AUDIO_CPU)

void EmulationReport()
{
#ifdef __WINS__
	FILE*	romFiles=fopen("c:\\romfiles.txt", "w+");
	if (romFiles)
	{
		const struct GameDriver**	game = drivers;
		while (*game)
		{
			if ((*game)->drv)
			{
				if (!((*game)->clone_of && !((*game)->clone_of->flags & NOT_A_DRIVER)))
				{
					fprintf(romFiles, "%s.zip\r\n", (*game)->name);
				}
			}
			game++;
		}
		fclose(romFiles);
	}


	
	FILE*	file=fopen("c:\\gamelist.txt", "w+");
	TInt	noOfGames = 0;
	TInt	noOfClones = 0;
	if (file)
	{
		fprintf(file, "**************************************\r\n");
		fprintf(file, "*                                    *\r\n");
		fprintf(file, "* Supported games                    *\r\n");
		fprintf(file, "*                                    *\r\n");
		fprintf(file, "**************************************\r\n");

		const struct GameDriver**	game = drivers;
		while (*game)
		{
			if ((*game)->drv)
			{
				noOfGames++;
				if ((*game)->clone_of && !((*game)->clone_of->flags & NOT_A_DRIVER))
					noOfClones++;
				char	name[64];
				sprintf(name, "(%s)", (*game)->name);

				float	freq = (*game)->drv->cpu[0].cpu_clock;
				freq /= (1000 * 1000);
		
				TBool soundSupported = ETrue;
				for (TInt i=0 ; i<MAX_CPU && soundSupported; i++)
				{
					TInt	cpuType = (*game)->drv->cpu[i].cpu_type;
					if (CPU_AUDIO(cpuType))
						soundSupported = EFalse;
				}

				if (strlen((*game)->description) > 39)
				{
					fprintf(file, "%s\r\n", (*game)->description);
					fprintf(file, "          %-10s, %s @ %f MHz, sound %s\r\n", name, cputype_name((*game)->drv->cpu[0].cpu_type), freq, soundSupported?"yes":"no");
				}
				else
				{
					fprintf(file, "%-40s %-10s %s @ %f MHz, sound %s\r\n", (*game)->description, name, cputype_name((*game)->drv->cpu[0].cpu_type), freq, soundSupported?"yes":"no");
				}
			}
			game++;
		}
		fprintf(file, "\r\n\r\n");
		fprintf(file, "  Unique games : %3d\r\n", noOfGames-noOfClones);
		fprintf(file, "  Clones       : %3d\r\n", noOfClones);
		fprintf(file, "  Total        : %3d\r\n", noOfGames);

		fprintf(file, "\r\n\r\n**************************************\r\n");
		fprintf(file, "*                                    *\r\n");
		fprintf(file, "* Supported cpus                     *\r\n");
		fprintf(file, "*                                    *\r\n");
		fprintf(file, "**************************************\r\n");

		TInt	cpuNum = 0;
		struct cpu_interface*	cpu = &cpuintf[0];
		while (cpu->cpu_num != -1)
		{
			if (cpu->cpu_num != 0)
			{
				fprintf(file, "%s v%s (%s) %s\r\n", 
							cputype_name(cpuNum),
							cputype_core_version(cpuNum),
							cputype_core_family(cpuNum),
							cputype_core_credits(cpuNum)
							);
			}
			cpu++;
			cpuNum++;
		}


		fprintf(file, "\r\n\r\n**************************************\r\n");
		fprintf(file, "*                                    *\r\n");
		fprintf(file, "* Supported soundhrdw                *\r\n");
		fprintf(file, "*                                    *\r\n");
		fprintf(file, "**************************************\r\n");

		struct snd_interface* sound = &sndintf[0];
		while (sound->sound_num != -1)
		{
			if (sound->start)
			{
				fprintf(file, "%s\r\n", sound->name);
			}
			sound++;
		}

		fclose(file);
	}
#endif
}


#define DRIVER_LIST_FILE_NAME	"c:\\driverlist.h"

void CheckGamesL()
{
#ifdef __WINS__

#ifdef __FULL_DRIVER_LIST__
		FILE*	file=fopen(DRIVER_LIST_FILE_NAME, "w+");
#endif

	const struct GameDriver**	game = drivers;
	while (*game)
	{
		TBool	gameSupported = ETrue;
		if ((*game)->drv)
		{
			for (TInt i=0 ; i<MAX_CPU ; i++)
			{
				TInt	cpuType = (*game)->drv->cpu[i].cpu_type;
				if (!CPU_AUDIO(cpuType) && (cpuType>0))
				{
					if (CPU_TYPE(cpuType) == 0)
					{
						RDebug::Print(HERE("%s: cpu type %d not supported"), (*game)->name, cpuType);
						gameSupported = EFalse;
					}
#ifdef __FULL_DRIVER_LIST__
						/*
						 * grmbl, I need to define cpu core M68000 to satisfy the link
						 * but I don't want them to show up in the driverlist.h 
						 */
					if (CPU_TYPE(cpuType) != 1)
						gameSupported = EFalse;
#endif
				}
			}

			for (i=0 ; i<MAX_SOUND ; i++)
			{
				TInt	soundType = (*game)->drv->sound[i].sound_type;
				if (soundType && !sndintf[soundType].start)
				{
					RDebug::Print(HERE("%s: sound type %d not supported"), (*game)->name, soundType);
					gameSupported = EFalse;
				}
			}

		}
#ifdef __FULL_DRIVER_LIST__
		if (gameSupported)
		{
			fprintf(file, "	DRIVER(%s)\n", (*game)->name);
		}
#endif

		game++;
	}
	
	EmulationReport();

#ifdef __FULL_DRIVER_LIST__
	fclose(file);
#endif

#endif
}


void CheckAudioSupport(TInt aGame)
{
	if (sGameOptions.iSoundEnabled)
	{
		TBool		soundSupported = ETrue;

		const struct GameDriver*	game = *(drivers + aGame);
		for (TInt i=0 ; i<MAX_CPU && soundSupported; i++)
		{
			TInt	cpuType = game->drv->cpu[i].cpu_type;
			if (CPU_AUDIO(cpuType))
				soundSupported = EFalse;
		}

		if (!soundSupported)
		{
			User::InfoPrint(_L("Sound is not supported in this game"));
			sGameOptions.iSoundEnabled = EFalse;
		}
	}
}

void RunGameL(TInt aGame)
{
	CheckAudioSupport(aGame);

	TInt err = run_game(aGame);
	
	RestoreGlobalSettings();		// restore contrast before displaying error
	if (err)
	{
			/*
			 * "Normal" game error usually because of roms not being present
			 */
		printf("Press any key to continue\n");
		getchar();
	}
}


void RunGame(TInt aGame, const TGameOptions& aGameOptions)
{
#ifdef __WINS__
	int argc=0;
	char** argv=0;
	char** envp=0;

	__crt0(argc,argv,envp);
#endif

	sGlobalData.iForcedExitFromFrontEnd = EFalse;
	SetDefaultGamesOptions();
	SetGamesOptions(aGameOptions);

	TRAPD(err, RunGameL(aGame));
	if (err)
	{
		Oops();
	}

	SendMameCloseEvent();

#ifdef __WINS__
	fclose(stdin);
	fclose(stdout);
	fclose(stderr);
#endif
}

void CreateGlobalDataL()
{
	sGlobalData.iGameDrivers = (struct GameDriver **) drivers;
	sGlobalData.iRunGameFunc = RunGame;
	sGlobalData.iCpunumNameFunc = cputype_name;
	sGlobalData.iShutdownFunc = ShutdownMame;

	User::LeaveIfError(sGlobalDataChunk.CreateGlobal(KGlobalDataChunkName, 4, 4));
	*((TUint32*) sGlobalDataChunk.Base()) = (TUint32) &sGlobalData;
}

void EpocMameL()
{
	CheckGamesL();
	CreateGlobalDataL();
}



#ifdef __WINS__

EXPORT_C TInt InitEmulator()
{
	CheckGamesL();
	CreateGlobalDataL();

	return KErrNone;
}


int __cdecl E32Dll(TDllReason aReason)
{
	return(KErrNone);
}

int main(int argc, char* argv[])
{
	DEBUGGER();
	return 0;
}



#else

#define APP_ON_C "C:\\EMAME\\APP\\EMAMEAPP.APP"
#define APP_ON_D "D:\\EMAME\\APP\\EMAMEAPP.APP"


void TargetEpocMameL()
{
	EpocMameL();

	CApaCommandLine* apaCommandLine = CApaCommandLine::NewLC();
	if (access(APP_ON_D, 0) != -1)
		apaCommandLine->SetLibraryNameL(_L(APP_ON_D));
	else
		apaCommandLine->SetLibraryNameL(_L(APP_ON_C));

	CleanupStack::Pop();
	EikDll::RunAppInsideThread(apaCommandLine);
}


int TargetMame()
{
	CTrapCleanup* cleanup=CTrapCleanup::New();

	TRAPD(error, TargetEpocMameL());

	__ASSERT_ALWAYS(!error, User::Panic(_L("EMAME"), error));

	delete cleanup;

	return 0;
}


extern "C" {
int main(int argc, char* argv[])
{
	TargetMame();

	return 0;
}
}		// extern "C"

#endif




