/*******************************************************************
 *
 *	File:		Audio.cpp
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

#include "audio.h"

#include <f32file.h>
#include <e32svr.h>


unsigned char linear2alaw(int pcm_val);


/*
 * In ER5 we can directly interface with the sound driver. In ER6 the preferred
 * way is to interace with the MediaServer. Apart from not having figured out
 * the MediaServer API, I suspect it will require client-server interaction.
 * Most likely that will be killing for EMame's performance. 
 *
 * I have taken yet another shortcut...
 * I noticed that in ER6 the sound driver's include files are not longer present
 * in \epoc32\include, however, the WINS sound driver binaries (esound.pdd and 
 * esdrv.ldd) are still there. I simply copied the ER5 include files over
 * to ER6, with the one change that in ER6 RDevSound should derive from 
 * RBusLogicalChannel rather than RLogicalChannel. If I then explicitly load 
 * the 2 sound drivers, sound works just fine in ER6/WINS. I haven't got a clue
 * on whether this will work on target.
 */


/*******************************************
 *
 * CMameAudio
 *
 *******************************************/


CMameAudio::CMameAudio(TBool aStereo, TInt aSamplesPerFrame) : iStereo(aStereo), iSamplesPerFrame(aSamplesPerFrame), iFrameCount(1)
{
}


CMameAudio* CMameAudio::NewL(TBool aStereo, TInt aSamplesPerFrame)
{
	CMameAudio*		self = new(ELeave) CMameAudio(aStereo, aSamplesPerFrame);
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();		// self
	return self;
}


CMameAudio::~CMameAudio()
{
	Destruct();
}

void CMameAudio::Destruct()
{
	if (iDevSound.Handle())
		iDevSound.Close();
}

#define KLogicalSoundDriverName _L("ESOUND")
#define KPhysicalSoundDriverName _L("ESDRV")

void CMameAudio::ConstructL()
{
#ifdef __ER6__
	User::LeaveIfError(User::LoadPhysicalDevice(KPhysicalSoundDriverName));
	User::LeaveIfError(User::LoadLogicalDevice(KLogicalSoundDriverName));
#endif

	User::LeaveIfError(iDevSound.Open());

	TSoundCapsV01			soundCaps;
	TPckg<TSoundCapsV01>	caps(soundCaps);

	iDevSound.Caps(caps);

	TSoundConfigV01			soundConfig;
	TPckg<TSoundConfigV01>	sc(soundConfig);

	iDevSound.Config(sc);
	soundConfig.iAlawBufferSize = 0x4000;
	soundConfig.iVolume = EVolumeMedium;
	User::LeaveIfError(iDevSound.SetConfig(sc));
	User::LeaveIfError(iDevSound.PreparePlayAlawBuffer());
}


TInt CMameAudio::FirstNoOfSamples()
{
	return iSamplesPerFrame;
}

void CMameAudio::SoundUpdate()
{
	if (iFrameCount > 8)
	{
		const TPtrC8		ptr((const TUint8*) iAlawSoundBuffer, iSamplesPerFrame *iFrameCount);
		if (iStatus == KRequestPending)
		{
			RDebug::Print(_L("Skipping sound"));
			iDevSound.PlayCancel();
		}
		iStatus = KRequestPending;
		iDevSound.PlayAlawData(iStatus, ptr);
//		User::WaitForRequest(iStatus);
		iFrameCount = 1;
	}
}


TInt CMameAudio::ProcessSoundSamples(TInt16* aBuffer)
{
	ASSERT(((iFrameCount+1)*iSamplesPerFrame)< MAX_SOUND_BUFFER);
	
	TUint8*		alawSoundbuffer = iAlawSoundBuffer + (iFrameCount*iSamplesPerFrame);
	for (TInt i=0 ; i<iSamplesPerFrame; i++)
	{
		*alawSoundbuffer++ = linear2alaw(*aBuffer++);
		if (iStereo)
			aBuffer++;
	}
	iFrameCount++;
	iTotalFrameCount++;

	return iSamplesPerFrame;
}




/******************************************************************************
 *
 * The following sound conversion routines have been copied and pasted
 * from the "spAudio - Audio I/O Library".
 *
 ******************************************************************************/
 
/*
 *
 * This source code is a product of Sun Microsystems, Inc. and is provided
 * for unrestricted use.  Users may copy or modify this source code without
 * charge.
 *
 * SUN SOURCE CODE IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING
 * THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun source code is provided with no support and without any obligation on
 * the part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY THIS SOFTWARE
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 * 
 *
 */

#define	SIGN_BIT	(0x80)		/* Sign bit for a A-law byte. */
#define	QUANT_MASK	(0xf)		/* Quantization field mask. */
#define	NSEGS		(8)		/* Number of A-law segments. */
#define	SEG_SHIFT	(4)		/* Left shift for segment number. */
#define	SEG_MASK	(0x70)		/* Segment field mask. */


static short seg_end[8] = {0xFF, 0x1FF, 0x3FF, 0x7FF,
			    0xFFF, 0x1FFF, 0x3FFF, 0x7FFF};

static int search(int val, short *table, int size)
{
	int		i;

	for (i = 0; i < size; i++) {
		if (val <= *table++)
			return (i);
	}
	return (size);
}

unsigned char linear2alaw(int pcm_val)	/* 2's complement (16-bit range) */
{
	int		mask;
	int		seg;
	unsigned char	aval;

	if (pcm_val >= 0) {
		mask = 0xD5;		/* sign (7th) bit = 1 */
	} else {
		mask = 0x55;		/* sign bit = 0 */
		pcm_val = -pcm_val - 8;
	}

	/* Convert the scaled magnitude to segment number. */
	seg = search(pcm_val, seg_end, 8);

	/* Combine the sign, segment, and quantization bits. */

	if (seg >= 8)		/* out of range, return maximum value. */
		return (unsigned char) (0x7F ^ mask);
	else {
		aval = (unsigned char) (seg << SEG_SHIFT);
		if (seg < 2)
			aval |= (pcm_val >> 4) & QUANT_MASK;
		else
			aval |= (pcm_val >> (seg + 3)) & QUANT_MASK;
		return (unsigned char) (aval ^ mask);
	}
}
