/*******************************************************************
 *
 *	File:		Video.h
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

#ifndef __VIDEO_H
#define __VIDEO_H

extern "C" {
#include "osdepend.h"
}

#include "globaldata.h"


#include <e32base.h>
#include <w32std.h>

#define MAX_ONSCREENCONTROLS	4
#define MAX_KEYPRESS			5

const TInt KSizeMameColorPalette = 256;

enum TOrientation
{
	EOrientationNormal,
	EOrientationRotLeft,
	EOrientationRotRight
};


class TOnScreenControlList
{
public:
	TInt	iKey;
	TRect	iRect;
};


class CMameFbsBitmap : public CFbsBitmap
{
public:
	void MameLockHeap() {CFbsBitmap::LockHeap(ETrue);}
	CBitwiseBitmap* MameAddress() const {return iAddressPointer;}
};


class CMameBitmap : public CBase
{
public:
	~CMameBitmap();
	static CMameBitmap* NewL(int aWidth, int aHeight, TDisplayMode aDisplayMode);

	struct osd_bitmap*	OsdBitmap() {return &iOsdBitmap;}
	struct osd_bitmap*	OsdVisibleBitmap() {return &iOsdVisibleBitmap;}
	CFbsBitmap*			FbsBitmap(){return iBitmap;}
	void				SetVisibleAreaL(const TRect& aVisibleArea);

	static TDisplayMode GetDisplayMode(TInt aDepth, TBool aColor = ETrue);
	static TInt			GetBpp(TDisplayMode aDisplayMode);
	static TInt			GetColorDepth(TDisplayMode aDisplayMode);

	
protected:
	CMameBitmap();
	void ConstructL(int aWidth, int aHeight, TDisplayMode aDisplayMode);

	struct osd_bitmap		iOsdBitmap;
	struct osd_bitmap		iOsdVisibleBitmap;

	CGraphicsContext*		iBitmapContext;
	CMameFbsBitmap*			iBitmap;
	CFbsBitmapDevice*		iBitmapDevice;
};


typedef RPointerArray<CMameBitmap>	CMameBitmapArray;
const TInt							KMameBitmapArrayGranularity = 5;

class CMameScreen : public CBase
{
public:
	CMameScreen(){};
	~CMameScreen();
	CMameScreen(const TScreenInfoV01& aScreenInfo, TDisplayMode aDisplayMode);
	virtual void ConstructL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps);

	void DrawBitmap(CMameBitmap* aMameBitmap){(this->*iBitmapDrawer)(aMameBitmap);}
	void AllocateColors(unsigned int totalcolors, const UINT8 *palette, UINT16 *pens, int modifiable, const UINT8 *debug_palette,UINT16 *debug_pens);

	TDisplayMode	DisplayMode(){return iDisplayMode;}
	TOrientation	Orientation(){return iOrientation;}
	TBool			HasFrameBuffer(){return iHasFrameBuffer;}
	TPoint			Position() {return iPosition;}
	TSize			ScreenSize() {return iScreenSize;}
	TSize			GameBitmapSize(){return TSize(iWidth, iHeight);}
	TBool			DoubleFrame(int aShowFps, struct osd_bitmap *bitmap);
	TInt			SkipThisFrame();

	void			SetWantScaling(TBool aWantScaling);
	void			SetWantDoubleBitmap(TBool aWantDoubleBitmap);

protected:
	typedef void (CMameScreen::*TScaledScanLineDrawer)(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge);
	TScaledScanLineDrawer	iScaledScanLineDrawer;

	typedef void (CMameScreen::*TBitmapDrawer)(CMameBitmap* aMameBitmap);
	TBitmapDrawer	iBitmapDrawer;

	void DrawScaledScanLine8bppTo2bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge);
	void DrawScaledScanLine8bppTo4bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge);
	void DrawScaledScanLine8bppTo8bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge);
	void DrawScaledScanLine8bppTo12bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge);

	void DrawDoubleScaledScanLine8bppTo8bpp(TUint* aSrcLine, TUint* aDstLine, TInt aWidth, TBool aMerge);

	void DirectDrawBitmap8bppTo2bpp(CMameBitmap* aMameBitmap);
	void DirectDrawBitmap8bppTo4bpp(CMameBitmap* aMameBitmap);
	void DirectDrawBitmap8bppTo8bpp(CMameBitmap* aMameBitmap);
	void DirectDrawBitmap8bppTo12bpp(CMameBitmap* aMameBitmap);
	void ScaledDrawBitmap(CMameBitmap* aMameBitmap);	// uses ScaledDrawScanLineXbppToYbpp

	void DirectDrawDoubleBitmap8bppTo8bpp(CMameBitmap* aMameBitmap);
	void DoubleScaledDrawBitmap(CMameBitmap* aMameBitmap);

	void	SetBitmapPosition(const TPoint& aPosition);
	void	SetupScreenAccess();
	void	SetupScaling();
	void	SetupPosition();

	TInt	AllocateMamePen(const TRgb& aColor);
	TUint	GetColorValue(const TRgb& aColor, const TDisplayMode& aDisplayMode);
	void	AllocateMameColors(TUint totalcolors, const UINT8 *palette, UINT16 *pens, TInt modifiable, const UINT8 *debug_palette,UINT16 *debug_pens);
	void	AllocateEpocColors(TUint totalcolors, const UINT8 *palette, UINT16 *pens, TInt modifiable, const UINT8 *debug_palette,UINT16 *debug_pens);

	TInt				iMamePaletteMaxIndex;
	TUint				iMamePalette[KSizeMameColorPalette];

	TUint8*				iScanLine;

	TBool				iHasFrameBuffer;
	TInt				iBytesPerBitmapLine;
	TInt				iBytesPerScanLine;
	TDisplayMode		iDisplayMode;
	TSize				iScreenSize;
	TUint8*				iBitmapStartAddress;

	TInt				iWidth;
	TInt				iHeight;
	TInt				iGameDepth;
	TInt				iGameWidth;
	TInt				iGameHeight;
	TInt				iPixelSkipX;
	TInt				iPixelSkipY;
	TPoint				iPosition;

	TBool				iWantPanning;
	TBool				iWantScaling;
	TBool				iWantDoubleBitmap;
	TBool				iDoubleBitmap;
	TBool				iWantMaintainRatioWhenScaling;
	TBool				iFirstTime;

	TOrientation		iOrientation;

	TPoint				iBitmapPosition;
	TUint8*				iFrameBuffer;		// if NULL we can't do direct screen access

/**************************************
 *
 * Frame skip + throttle
 *
 **************************************/
	TInt		iVideoFps;
	TInt		iPeriod;
	TInt		iDoubleFrame;
	TInt		iDoubleFrameCount;
	TInt		iRealFrameCount;
	TInt		iGameFrameCount;
	TInt		iPrevTicks;

	TInt		iTicksGameStart;
	TInt		iTotalRealFrames;
	TInt		iTotalGameFrames;

	TInt		iDiffFrames;
	TInt		iRealFps;
	TInt		iGameFps;
};


#ifdef __WINS__

	/*
	 * Sothat we can test the target drawing routines under target
	 */
class CWinsMameScreen : public CMameScreen
{
public:
	CWinsMameScreen(){}
	CWinsMameScreen(const TSize& aScreenSize, TDisplayMode aDisplayMode);
	~CWinsMameScreen();
	virtual void ConstructL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps);
	CMameBitmap*	iEmulatedScreenBitmap;
};
#endif


class CMameWindow : public CBase
{
public:
	~CMameWindow();
	static CMameWindow* NewL();

	void CreateDisplayL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps, const TGameOptions& aGameOptions);
	void CloseDisplay();
	void SetVisbibleAreaL(const TRect aRect);

	CMameBitmap*	AllocateBitmapL(TInt aWidth, TInt aHeight, TInt aDepth);
	void DrawBitmap(CMameBitmap* aMameBitmap);

	void MarkInvalid(TInt aXmin, TInt aYmin, TInt aXmax, TInt aYmax);
	TInt SkipThisFrame();

	void AllocateColor(unsigned int totalcolors, const UINT8 *palette, UINT16 *pens, int modifiable, const UINT8 *debug_palette,UINT16 *debug_pens);

	TBool IsKeyPressed(TInt aKey);
	void Pause(TBool aPause);

protected:
	CMameWindow();
	void ConstructL();
	void Destruct();

	void HandleWsEvent(const TWsEvent& aWsEvent);
	void HandleRedrawEvent(const TWsRedrawEvent& iRedrawEvent);
	void StartWServEvents();
	void PollWServEvents();
	void CancelWServEvents();
	void EmulateTargetDrawBitmap(CMameBitmap* aMameBitmap);
	void WsBlitBitmap(CMameBitmap* aMameBitmap);
	void DrawOnScreenControlsNow();

	TGameOptions			iGameOptions;

	CMameBitmapArray*		iMameBitmapArray;

	TBool					iShowFps;

	CMameScreen*		iMameScreen;
	void				CreateMameScreenL(TInt aWidth, TInt aHeight, TInt aDepth, TOrientation aOrientation, TInt aFps);

	TInt				iLastOnScreenKey;
	TBool				iIsPaused;

	TOnScreenControlList	iOnScreenControlList[MAX_ONSCREENCONTROLS];
	void SetupOnScreenControls();
	void DrawOnScreenControls();
	TInt GetOnScreenKey(const TPoint aPosition);
	void HandlePointerEventL(const TPointerEvent& aPointerEvent);

		
		/*
		 * Maintain state about key presses. A key is pressed from the moment we get
		 * its keydown event until we get its keyup event. While holding down a key
		 * another key may be pressed. Hence we use this array below to keep track
		 * of these nested key presses.
		 */
	struct TKeyPress
	{
		TInt	iScanCode;
		TInt	iKeyCode;
	};

	TKeyPress	iKeyPress[MAX_KEYPRESS];
	TInt		iKeyPressLevel;
	void KeyDownEvent(TInt aScanCode);
	void KeyPressEvent(TInt aKeyCode);
	void KeyUpEvent(TInt aScanCode);
	TInt CurrentKey();



		/*
		 * Window Server stuff
		 */
	RWsSession			iWsSession;
	RWindowGroup		iWsWindowGroup;
	RWindow				iWsWindow;
	CWsScreenDevice*	iWsScreen;
	CWindowGc*			iWindowGc;

	TRequestStatus		iWsEventStatus;
	TRequestStatus		iRedrawEventStatus;
	TWsEvent			iWsEvent;
	TWsRedrawEvent		iRedrawEvent;
};



#endif			/* __VIDEO_H */
