/*******************************************************************
 *
 *	File:		EMameApp.cpp
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

#include <eikenv.h>
#include <eikdialg.h>
#include <eiktxlbx.h>
#include <eiktxlbm.h>
#include <eikdlgtb.h>
#include <eiklabel.h>
#include <eikchlst.h>
#include <eikchkbx.h>
#ifndef __ER6__
#include <eikdialg.hrh>
#endif

#include "emameapp.h"
#include "emameapp.rsg"
#include "..\group\emameapp.mbg"
#include "version.h"
#include "AppPlatform.h"


/************************************************
 *
 * GameRunning Dialog
 *
 ************************************************/

class CGameRunningDialog : public CEikDialog
{
public:
	CGameRunningDialog(const TDesC8& aGameName, CMameEngine& aMameEngine);
protected: // framework
    void PreLayoutDynInitL();
	void SetSizeAndPositionL(const TSize& aSize);
	void HandleControlStateChangeL(TInt aControlId);

	TBuf<64>			iGameName;
	CMameEngine&		iMameEngine;
};

CGameRunningDialog::CGameRunningDialog(const TDesC8& aGameName, CMameEngine& aMameEngine) : iMameEngine(aMameEngine)
{
	iGameName.Copy(aGameName);
}

void CGameRunningDialog::PreLayoutDynInitL()
{
	CEikLabel*	label =(CEikLabel*)Control(ECtlRunningGameName);
	label->SetTextL(iGameName);
}

void CGameRunningDialog::HandleControlStateChangeL(TInt aControlId)
{
	switch (aControlId)
	{
		case ECtlCmdButPanelRunningGame:
		{
			const TInt ctrlId=STATIC_CAST(CEikDialogToolBar*, Control(aControlId))->StateChangeControlId();
			switch (ctrlId)
			{
				case ECtlCmdButStopGame:
					iMameEngine.ForceGameExit();
					break;
				default:
					break;
			}
			break;
		}
		default:
			break;
	}
}

void CGameRunningDialog::SetSizeAndPositionL(const TSize& aSize)
{
	SET_CORNER_AND_SIZE(EHCenterVCenter, aSize);
}


/************************************************
 *
 * About Dialog
 *
 ************************************************/


class CAboutDialog : public CEikDialog
{
public:
	CAboutDialog(){}
protected: // framework
    void PreLayoutDynInitL();
	void SetSizeAndPositionL(const TSize& aSize);
};

void CAboutDialog::PreLayoutDynInitL()
{
	TBuf<128>	text;
	TBuf<16>	versionText;

#if (EMAME_VERSION_PATCH > 0)
	versionText.Format(_L("%d.%d.%d"), EMAME_VERSION_MAJOR, EMAME_VERSION_MINOR, EMAME_VERSION_PATCH);
#else
	versionText.Format(_L("%d.%d"), EMAME_VERSION_MAJOR, EMAME_VERSION_MINOR);
#endif

	HBufC*		versionFormat =	CEikonEnv::Static()->AllocReadResourceL(R_ABOUT_VERSION_TEXT);
	CleanupStack::PushL(versionFormat);

	const TDesC&	versionFormatDes = versionFormat->Des();
	text.Format(versionFormatDes, &versionText);

	CEikLabel*	label =(CEikLabel*)Control(ECtlAboutVersion);
	label->SetTextL(text);

	CleanupStack::PopAndDestroy(); // 	versionFormat;
}

void CAboutDialog::SetSizeAndPositionL(const TSize& aSize)
{
//	TSize	size(400, 200);
	SET_CORNER_AND_SIZE(EHCenterVCenter, aSize);
}



/************************************************
 *
 * Help Dialog
 *
 ************************************************/

class CMameHelpDialog : public CEikDialog
{
public:
	CMameHelpDialog(){}
protected: // framework
    void PreLayoutDynInitL();
	void SetSizeAndPositionL(const TSize& aSize);
};

void CMameHelpDialog::PreLayoutDynInitL()
{
}

void CMameHelpDialog::SetSizeAndPositionL(const TSize& aSize)
{
//	TSize	size(400, 200);
	SET_CORNER_AND_SIZE(EHCenterVCenter, aSize);
}


/************************************************
 *
 * Game Info Dialog
 *
 ************************************************/

class CGameInfoDialog : public CEikDialog
{
public:
	CGameInfoDialog(CMameEngine& aMameEngine, TInt aGame) : iMameEngine(aMameEngine), iGame(aGame){}
protected: // framework
    void PreLayoutDynInitL();
	void SetSizeAndPositionL(const TSize& aSize);
	CMameEngine&		iMameEngine;
	TInt				iGame;
};

void CGameInfoDialog::PreLayoutDynInitL()
{
	TBuf8<128>	text8;
	TBuf<128>	text;
	CEikLabel*	label;
	
	struct GameDriver*	game = *(iMameEngine.GameDrivers() + iGame);

	label =(CEikLabel*)Control(ECtlGameInfoGameDescription);
	text8.Format(_L8("%s [%s]"), game->description, (const TUint8*)game->name);
	text.Copy(text8);
	label->SetTextL(text);

	if (iMameEngine.IsGameCloneOf(game))
	{
		label =(CEikLabel*)Control(ECtlGameInfoCloneOff);
		text8.Format(_L8("%s"), game->clone_of->description);
		text.Copy(text8);
		label->SetTextL(text);
	}
	else
	{
		DeleteLine(ECtlGameInfoCloneOff);
	}

	label =(CEikLabel*)Control(ECtlGameInfoGameManufacturer);
	text8.Format(_L8("%s, %s"), game->year, game->manufacturer);
	text.Copy(text8);
	label->SetTextL(text);

	label =(CEikLabel*)Control(ECtlGameInfoGameResolution);
	text8.Format(_L8("%d x %d, using %d colors"), game->drv->screen_width, game->drv->screen_height, game->drv->total_colors);
	text.Copy(text8);
	label->SetTextL(text);

	label =(CEikLabel*)Control(ECtlGameInfoGameCpu);
	float	freq = game->drv->cpu[0].cpu_clock;
	freq /= (1000 * 1000);
	text8.Format(_L8("%s @ %f MHz"), iMameEngine.CpuName(game->drv->cpu[0].cpu_type), freq);
	text.Copy(text8);
	label->SetTextL(text);
}


void CGameInfoDialog::SetSizeAndPositionL(const TSize& aSize)
{
	TSize size = aSize;
	if (size.iWidth < 400)
		size.iWidth = 400;
	SET_CORNER_AND_SIZE(EHCenterVCenter, size);
}



/************************************************
 *
 * Game Options Dialog
 *
 ************************************************/


class CGameOptionsDialog : public CEikDialog
{
public:
	CGameOptionsDialog(TGameOptions& aGameOptions, TBool aIsColorDevice) : iGameOptions(aGameOptions), iIsColorDevice(aIsColorDevice){}
protected: // framework
    void PreLayoutDynInitL();
	void SetSizeAndPositionL(const TSize& aSize);
    TBool OkToExitL(TInt aKeycode);
	TGameOptions&		iGameOptions;
	TBool				iIsColorDevice;
};


void CGameOptionsDialog::PreLayoutDynInitL()
{
	CEikChoiceList* choiceList;

	CEikCheckBox* checkBox=(CEikCheckBox*)Control(ECtlGameOptionsSound);
	checkBox->SetState(iGameOptions.iSoundEnabled ? CEikButtonBase::ESet : CEikButtonBase::EClear);

	if (iIsColorDevice)
	{
		choiceList=(CEikChoiceList*)Control(ECtlGameOptionsGameSize);
		choiceList->SetCurrentItem((TInt) iGameOptions.iGameSize);

		DeleteLine(ECtlGameOptionsOrientation);
		DeleteLine(ECtlGameOptionsContrast);
	}
	else
	{
		choiceList=(CEikChoiceList*)Control(ECtlGameOptionsOrientation);
		choiceList->SetCurrentItem((TInt) iGameOptions.iOrientation);

		choiceList=(CEikChoiceList*)Control(ECtlGameOptionsContrast);
		choiceList->SetCurrentItem((TInt) iGameOptions.iContrast);

		DeleteLine(ECtlGameOptionsGameSize);
	}
}



void CGameOptionsDialog::SetSizeAndPositionL(const TSize& aSize)
{
	TSize size = aSize;
	if (size.iWidth < 400)
		size.iWidth = 400;
	SET_CORNER_AND_SIZE(EHCenterVCenter, size);
}

    
TBool CGameOptionsDialog::OkToExitL(TInt aKeycode)
{
	CEikChoiceList* choiceList;

	CEikCheckBox* checkBox=(CEikCheckBox*)Control(ECtlGameOptionsSound);
	iGameOptions.iSoundEnabled = (checkBox->State() == CEikButtonBase::ESet) ? (TBool) ETrue : (TBool) EFalse;

	if (iIsColorDevice)
	{
		choiceList=(CEikChoiceList*)Control(ECtlGameOptionsGameSize);
		iGameOptions.iGameSize = (TGameOptions::TGameSize) choiceList->CurrentItem();
	}
	else
	{
		choiceList=(CEikChoiceList*)Control(ECtlGameOptionsOrientation);
		iGameOptions.iOrientation = (TGameOptions::TOrientation) choiceList->CurrentItem();

		choiceList=(CEikChoiceList*)Control(ECtlGameOptionsContrast);
		iGameOptions.iContrast = (TGameOptions::TContrast) choiceList->CurrentItem();
	}

	return ETrue;
}


/************************************************
 *
 * Play List Dialog
 *
 ************************************************/

class CPlayListDialog : public CEikDialog
{
public:
	CPlayListDialog(CMameEngine& aMameEngine, TInt& aGameToRun) : iMameEngine(aMameEngine), iGameToRun(aGameToRun){}
protected: // framework
    void PreLayoutDynInitL();
	void HandleControlStateChangeL(TInt aControlId);
	void SetSizeAndPositionL(const TSize& aSize);
	void ShowGameDetailsL(TInt aGame);
	TInt GameIndex();

    TBool OkToExitL(TInt aKeycode);
	CMameEngine&		iMameEngine;
	TInt&				iGameToRun;
};

void CPlayListDialog::SetSizeAndPositionL(const TSize& aSize)
{
	TSize dialogSize(400, aSize.iHeight);
	SET_CORNER_AND_SIZE(EHCenterVCenter,dialogSize);
}


void CPlayListDialog::ShowGameDetailsL(TInt aGame)
{
	CGameInfoDialog*	dialog = new CGameInfoDialog(iMameEngine, aGame);
	dialog->ExecuteLD(R_DIALOG_GAME_INFO);
}

void CPlayListDialog::HandleControlStateChangeL(TInt aControlId)
{
	switch(aControlId)
	{
		case ECtlCmdButPanel:
		{
			const TInt ctrlId=STATIC_CAST(CEikDialogToolBar*, Control(aControlId))->StateChangeControlId();
			switch (ctrlId)
			{
				case ECtlCmdButRun:
					TryExitL(EEikBidOk);
					break;
				case ECtlCmdButCancel:
					TryExitL(EEikBidCancel);
					break;
				case ECtlCmdButInfo:
					ShowGameDetailsL(GameIndex());
					break;
			}
			break;
		}
	}
}

void CPlayListDialog::PreLayoutDynInitL()
{
	TBuf8<128>	gameName8;
	TBuf<128>	gameName;

	CEikTextListBox* listbox=(CEikTextListBox*)Control(ECtlListboxPlayList);
	CDesCArray* listboxArray=((CDesCArray*)listbox->Model()->ItemTextArray());

	// append contents of the model's checksum array to listbox for display
	struct GameDriver**	game = iMameEngine.GameDrivers();
	while(*game)
	{
		gameName8.Format(_L8("%s"), (*game)->description);
		gameName.Copy(gameName8);
		listboxArray->AppendL(gameName);
		game++;
	}
	listboxArray->Sort();

	if (iGameToRun != -1)
	{
		TInt	index = 0;
		gameName8.Format(_L8("%s"), (*(iMameEngine.GameDrivers() + iGameToRun))->description);
		gameName.Copy(gameName8);
		if (listboxArray->FindIsq(gameName, index) == KErrNone)
			listbox->SetCurrentItemIndex(index);
	}

	CEikScrollBarFrame* scrollbarFrame = listbox->CreateScrollBarFrameL();
	// set horizontal and vertical scroll bars to appear when required
	scrollbarFrame->SetScrollBarVisibilityL(CEikScrollBarFrame::EAuto,
		CEikScrollBarFrame::EAuto);
}

TInt CPlayListDialog::GameIndex()
{
	CEikTextListBox* listbox=(CEikTextListBox*)Control(ECtlListboxPlayList);
	CDesCArray* listboxArray=((CDesCArray*)listbox->Model()->ItemTextArray());
	return iMameEngine.GameNo(listboxArray->operator[](listbox->CurrentItemIndex()));
}

TBool CPlayListDialog::OkToExitL(TInt /*aKeycode*/)
{
	TBuf<128>	gameName;
	iGameToRun = GameIndex();
    return(ETrue);
}








CApaDocument* CMameApplication::CreateDocumentL()
{
	CMameDocument*	doc = new (ELeave) CMameDocument(*this);

	CleanupStack::PushL(doc);
	doc->ConstructL();
	CleanupStack::Pop();

	return doc;
}



CMameDocument::CMameDocument(CEikApplication& aApp)
		: CEikDocument(aApp)
{
}

CMameDocument::~CMameDocument()
{
	delete iMameEngine;
}

void CMameDocument::ConstructL()
{
	iMameEngine = CMameEngine::NewL();
}


CEikAppUi* CMameDocument::CreateAppUiL()
{
	return new(ELeave) CMameAppUi;
}


CMameAppUi::~CMameAppUi()
{
	delete iAppView;
	delete iGameRunner;
}


void CMameAppUi::ConstructL()
{
	BaseConstructL();
	iAppView=new(ELeave) CMameAppView;
	iAppView->ConstructL(ClientRect());

	MameEngine().SetWindowGroupId(CEikonEnv::Static()->RootWin().Identifier());

	SetDefaultGameOptions(IsColorDevice());

	iGameRunner = CGameRunner::NewL(*this, MameEngine());
	DisplayGamePlayDialogL();
}

void CMameAppUi::SetDefaultGameOptions(TBool aIsColorDevice)
{
	TGameOptions gameOptions = MameEngine().GameOptions();

	gameOptions.iSoundEnabled = EFalse;
	
	if (aIsColorDevice)
	{
		gameOptions.iGameSize = TGameOptions::EGameSizeNormal;
		gameOptions.iSoundEnabled = ETrue;
	}
	else
	{
		gameOptions.iOrientation = TGameOptions::EOrientationLeft;
		gameOptions.iContrast = TGameOptions::EContrastSetToLow;
	}
	MameEngine().SetGameOptions(gameOptions);
}



TBool CMameAppUi::IsColorDevice()
{
	TDisplayMode	displayMode = CEikonEnv::Static()->ScreenDevice()->DisplayMode();

	return (displayMode >= EColor16);
}

void CMameAppUi::HandleApplicationSpecificEventL(TInt aType, const TWsEvent& aEvent)
{
	switch (aType)
	{
		case EMameShuttingDown:
			CEikonEnv::Static()->RootWin().SetOrdinalPosition(0);
//			MameEngine().ShutdownGame();
			StopGame();
//			DisplayGamePlayDialogL();		// put up dialog again
			break;

		default:
			break;
	}
}


void CMameAppUi::DisplayHelpDialogL()
{
	CEikDialog* dialog = new CMameHelpDialog();

	dialog->ExecuteLD(R_DIALOG_DIPLAY_EMAME_HELP);
}


void CMameAppUi::HandleCommandL(TInt aCommand)
{
	switch (aCommand)
	{
		case EEikCmdExit: 
			Exit();
			break;

		case EEikCmdGamePlay: 
			DisplayGamePlayDialogL();
			break;

		case EEikCmdGameOptions:
			DisplayGameOptionsDialogL();
			break;

		case EEikCmdHelpPlayEmame:
			DisplayHelpDialogL();
			break;

		case EEikCmdHelpAbout:
			DisplayAboutDialogL();
			break;
	}
}




void CMameAppUi::DisplayGameOptionsDialogL()
{
	TGameOptions	gameOptions = MameEngine().GameOptions();
	CEikDialog* dialog = new CGameOptionsDialog(gameOptions, IsColorDevice());

	if (dialog->ExecuteLD(R_DIALOG_GAME_OPTIONS))
	{
		MameEngine().SetGameOptions(gameOptions);
	}
}


void CMameAppUi::DisplayGamePlayDialogL()
{
	CEikDialog*	dialog = new CPlayListDialog(MameEngine(), iGameToRun);
	if (dialog->ExecuteLD(R_DIALOG_PLAY_LIST))
	{
		RunGameL();
	}
}


void CMameAppUi::RunGameL()
{
	TBuf8<64>	gameName;
	iGameRunner->RunGameL(iGameToRun);
	
	gameName.Format(_L8("%s"), MameEngine().GameName(iGameToRun));
	iGameRunningDialog = new CGameRunningDialog(gameName, MameEngine());
	iGameRunningDialog->ExecuteLD(R_DIALOG_RUNNING_GAME);
	
	DisplayGamePlayDialogL();
}


void CMameAppUi::StopGame()
{
	iGameRunner->TerminateGame();
	if (iGameRunningDialog)
	{
		delete iGameRunningDialog;
		iGameRunningDialog = NULL;
	}
}

void CMameAppUi::NotifyGameFinished()
{
	StopGame();
}


void CMameAppUi::DisplayAboutDialogL()
{
	CEikDialog*	dialog = new CAboutDialog;
	dialog->ExecuteLD(R_DIALOG_ABOUT);
}



void CMameAppView::ConstructL(const TRect& aRect)
{
	CreateWindowL();
	SET_RECT(aRect);
	ActivateL();

		/*
		 * Load background image; either grayscale or color depending on the device
		 * we're running on.
		 */
	TInt	image = CMameAppUi::Static()->IsColorDevice() ? EMbmEmameappEmame256 : EMbmEmameappEmameg16;
	TRAPD(err, iBgImage = CEikonEnv::Static()->CreateBitmapL(_L("c:\\emame\\app\\emameapp.mbm"), image));
	if (err)
	{
		TRAP(err, iBgImage = CEikonEnv::Static()->CreateBitmapL(_L("d:\\emame\\app\\emameapp.mbm"), image));
	}
	if (iBgImage)
	{
		iImagePosition.iX = (aRect.Size().iWidth - iBgImage->SizeInPixels().iWidth) / 2;
		iImagePosition.iY = (aRect.Size().iHeight - iBgImage->SizeInPixels().iHeight) / 2;
	}
}

CMameAppView::~CMameAppView()
{
	delete iBgImage;
}


void CMameAppView::Draw(const TRect& /*aRect*/) const
{
	CWindowGc& gc = SystemGc();

	if (iBgImage)
		gc.DrawBitmap(iImagePosition, iBgImage);
}


GLDEF_C TInt E32Dll(TDllReason)
{
	return KErrNone;
}

EXPORT_C CApaApplication* NewApplication()
{
	return new CMameApplication;
}



TUid CMameApplication::AppDllUid() const
{
	return KUidEmameApp;
}
